/*  EQEMu:  Everquest Server Emulator
	Copyright (C) 2001-2003  EQEMu Development Team (http://eqemulator.net)

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY except by those people which sell it, which
	are required to give you total support for your newly bought product;
	without even the implied warranty of MERCHANTABILITY or FITNESS FOR
	A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef EQ_PACKET_STRUCTS_H
#define EQ_PACKET_STRUCTS_H 

#include "types.h"
#include <string.h>
#include <time.h>

#define BUFF_COUNT 15

/*
** Item packet types
**
*/
enum ItemPacketType
{
	ItemPacketViewLink			= 0x00,
	ItemPacketTradeView			= 0x65,
	ItemPacketLoot				= 0x66,
	ItemPacketTrade				= 0x67,
	ItemPacketCharInventory		= 0x69,
	ItemPacketSummonItem		= 0x6A,
	ItemPacketMerchant			= 0x64,
	ItemPacketWorldContainer	= 0x6B
};

/*
** Item attributes
**
*/
enum ItemAttrib
{
	ItemAttribLore			= (1 << 0),
	ItemAttribArtifact		= (1 << 1),
	ItemAttribSummoned		= (1 << 2),
	ItemAttribMagic			= (1 << 3),
	ItemAttribAugment		= (1 << 4),
	ItemAttribPendingLore	= (1 << 5),
	ItemAttribNone			= 0,
	ItemAttribUnknown		= 0xFFFFFFFF
};

/*
** Item types
**
*/
enum ItemType
{
	ItemTypeCommon		= 0,
	ItemTypeContainer	= 1,
	ItemTypeBook		= 2
};

/*
** Compiler override to ensure
** byte aligned structures
*/
#pragma pack(1)
//#define INVERSEXY = 1

/* Name Approval Struct */
/* Len: */
/* Opcode: 0x8B20*/
struct NameApproval
{
	char name[64];
	int32 race;
	int32 class_;
	int32 deity;
};

/*
** Entity identification struct
** Size: 4 bytes
** OPCodes: OP_DeleteSpawn, OP_Assist
*/
struct EntityId_Struct
{
/*00*/	uint32	entity_id;
/*04*/
};

struct Duel_Struct
{
	int32 duel_initiator;
	int32 duel_target;
};

struct DuelResponse_Struct
{
	int32 target_id;
	int32 entity_id;
	int32 unknown;
};

/*
** Color_Struct
** Size: 4 bytes
** Used for convenience
** Merth: Gave struct a name so gcc 2.96 would compile
**
*/
struct Color_Struct
{
	union
	{
		struct
		{
			int8	blue;
			int8	green;
			int8	red;
			uint8	unused;
		} rgb;
		uint32 color;
	};
};

/*
** Character Selection Struct
** Length: 1660 Bytes
**
*/
struct CharacterSelect_Struct {
/*0000*/	char	name[10][64];		// Characters Names
/*0640*/	int8	level[10];			// Characters Levels
/*0650*/	int8	class_[10];			// Characters Classes
/*0660*/	int32	race[10];			// Characters Race
/*0700*/	int32	zone[10];			// Characters Current Zone
/*0740*/	int8	gender[10];			// Characters Gender
/*0750*/	int8	face[10];			// Characters Face Type
/*0760*/	int32	equip[10][9];		// 0=helm, 1=chest, 2=arm, 3=bracer, 4=hand, 5=leg, 6=boot, 7=melee1, 8=melee2
/*1120*/	Color_Struct	cs_colors[10][9];	// Characters Equipment Colors (RR GG BB 00)
/*1480*/	int32	deity[10];			// Characters Deity
/*1520*/	int32	melee[2][10];		// Characters primary and secondary IDFile number
/*1600*/	int8	unknown1600[60];	// ***Placeholder
};

/*
** Server Zone Entry Struct
** Length: 452 Bytes
** OPCodes: OP_ServerZoneEntry
** Fields from old struct not yet found:
**	int8	title;				// AA Title
**	int8	skyradius;			// Skyradius
**	float	delta_x
**	float	delta_y
**	float	delta_z
**	float	heading
**	float	delta_heading
**
*/
struct ServerZoneEntry_Struct
{
/*0000*/	uint32	checksum;			// Checksum for packet
/*0004*/	uint8	gm;					// Gm flag 0/1
/*0005*/	char	name[64];			// Character name
/*0069*/	char	last_name[32];		// Character last name
/*0101*/	char	unknown0069[7];		//
/*0108*/	float	y;					// position
/*0112*/	float	x;					// position
/*0116*/	float	z;					// position
/*0120*/	float	unknown0088;		//
/*0124*/	float	unknown0092;		//
/*0128*/	float	unknown0096;		//
/*0132*/	char	unknown0100[4];		//
/*0136*/	float	heading;			//
/*0140*/	float	unknown0108;		//
/*0144*/	char	unknown0112[88];	//
/*0232*/	uint8	walk_mode;			// zero=not walking, nonzero=walking (?)
/*0233*/	uint8	linkdead;			// LD=1, Not LD=0
/*0234*/	char	unknown0202[1];		//
/*0235*/	uint8	lfg;				// LFG=1, Not LFG=0
/*0236*/	char	unknown0204[4];		//
	union {
		struct {
		/*0240*/	Color_Struct color_helmet;		// Color of helmet item
		/*0244*/	Color_Struct color_chest;		// Color of chest item
		/*0248*/	Color_Struct color_arms;		// Color of arms item
		/*0252*/	Color_Struct color_bracers;		// Color of bracers item
		/*0256*/	Color_Struct color_hands;		// Color of hands item
		/*0260*/	Color_Struct color_legs;		// Color of legs item
		/*0264*/	Color_Struct color_feet;		// Color of feet item
		/*0268*/	Color_Struct color_primary;		// Color of primary item
		/*0272*/	Color_Struct color_secondary;	// Color of secondary item
		} equipment_colors;
		/*0240*/	Color_Struct colors[9];				// Array elements correspond to struct equipment_colors above
	};
	union {
		struct {
		/*0276*/	uint32	equip_helmet;		// Equipment: Helmet Visual
		/*0280*/	uint32	equip_chest;		// Equipment: Chest Visual
		/*0284*/	uint32	equip_arms;			// Equipment: Arms Visual
		/*0288*/	uint32	equip_bracers;		// Equipment: Bracers Visual
		/*0292*/	uint32	equip_hands;		// Equipment: Hands Visual
		/*0296*/	uint32	equip_legs;			// Equipment: Legs Visual
		/*0300*/	uint32	equip_feet;			// Equipment: Feet Visual
		/*0304*/	uint32	equip_primary;		// Equipment: Primary Visual
		/*0308*/	uint32	equip_secondary;	// Equipment: Secondary Visual
		} equipment;
		/*0276*/	uint32	equip[9];			// Array elements correspond to struct equipment above
	};
/*0312*/	uint16	zone_id;			// Current zone
/*0314*/	uint16	unknown0282;
/*0316*/	uint32	unknown0284;		// Looks like random garbage
										// @bp looks like a timer of sorts running backwards
										// Key auth time remaining?
/*0320*/	char	unknown0288[20];	// @bp Looks semi static
/*0340*/	float	unknown0340[2];		// @bp These don't seem to change
										// They control movement
/*0348*/	float	runspeed;			// Speed when running
/*0352*/	float	unknown0352[2];		//
/*0360*/	float	size;				// Size of character
										// Changing size works, but then movement stops!
/*0364*/	float	walkspeed;			// Speed when walking
/*0368*/	char	unknown0368[1];		//
/*0369*/	uint8	haircolor;			// Hair color
/*0370*/	uint8	beardcolor;			// Beard color
/*0371*/	uint8	eyecolor1;			// Left eye color
/*0372*/	uint8	eyecolor2;			// Right eye color
/*0373*/	uint8	hairstyle;			// Hair style
/*0374*/	uint8	beard;				// Beard type
/*0375*/	uint8	face;				// Face type // this may be element sze[321] also, not sure..
/*0376*/	uint8	level;				// Level
/*0377*/	char	unknown0377[1];		//
/*0378*/	uint8	gender;				// 0=male, 1=female, 2=neutral
/*0379*/	uint8	pvp;				// PVP=1, No PVP=0
/*0380*/	uint8	invis;				// 1=invis, other=not invis (not sure about this..)
/*0381*/	char	unknown0381[1];		//
/*0382*/	uint8	class_;				// Class
/*0383*/	char	unknown0383[13];	//
/*0396*/	uint32	race;				// Race
/*0400*/	uint8	anon;				// 0=non anon, 1=anon, 2=role, 3=role/anon
/*0401*/	char	unknown0401[27];	//
/*0428*/	uint32	deity;				// Deity
/*0432*/	char	unknown0432[4];		//
/*0436*/	uint32	guild_id;			// Current guild
/*0440*/	char	unknown0440[12];
/*0452*/
};

/*
** New Zone Struct
** Length: 576 Bytes
** Last Updated: Jul-04-2003
*/
struct NewZone_Struct {
/*0000*/	char	char_name[64];			// Character Name
/*0064*/	char	zone_short_name[32];	// Zone Short Name
/*0096*/	char	zone_long_name[278];	// Zone Long Name
/*0310*/	uint8	ztype;					// Zone type (usually FF)
/*0311*/	uint8	fog_red[4];				// Zone fog (red)
/*0315*/	uint8	fog_green[4];			// Zone fog (green)
/*0319*/	uint8	fog_blue[4];			// Zone fog (blue)
/*0374*/	uint8	unknown230[87];			// Unknown
/*0474*/	uint8	sky;					// Sky Type
/*0475*/	uint8	unknown331[13];			// ***Placeholder
/*0488*/	float	zone_exp_multiplier;	// Experience Multiplier
/*0492*/	float	safe_x;					// Zone Safe X (Not Inversed)
/*0496*/	float	safe_y;					// Zone Safe Y (Not Inversed)
/*0500*/	float	safe_z;					// Zone Safe Z
/*0504*/	float	unknown0504;			// ***Placeholder
/*0508*/	float	underworld;				// Underworld (Not Sure?)
/*0512*/	float	minclip;
	// Minimum View Distance
/*0516*/	float	maxclip;				// Maximum View DIstance
/*0520*/	int8	unknown_end[68];		// ***Placeholder
/*0588*/
};

struct CFGNewZone_Struct {
/*0000*/	char	char_name[64];			// Character Name
/*0064*/	char	zone_short_name[32];	// Zone Short Name
/*0096*/	char	zone_long_name[278];	// Zone Long Name
/*0278*/	int8	unknown230[100];		// ***Placeholder
/*0378*/	int8	sky;		  			// Sky Type
/*0379*/	int8	unknown331[9];			// ***Placeholder
/*0388*/	float	zone_exp_multiplier;	// Experience Multiplier
/*0392*/	float	safe_x;					// Zone Safe X (Not Inversed)
/*0396*/	float	safe_y;					// Zone Safe Y (Not Inversed)
/*0400*/	float	safe_z;					// Zone Safe Z
/*0404*/	float	unknown256;				// ***Placeholder
/*0408*/	float	underworld;				// Underworld (Not Sure?)
/*0412*/	float	minclip;				// Minimum View Distance
/*0416*/	float	maxclip;				// Maximum View DIstance
/*0420*/	int8	unknown_end[56];		// ***Placeholder
};

struct Discipline_Struct {
/*0000*/	int8 unknown0000[4];
/*0004*/	char  charname[64];
/*0068*/	char  charname2[64];
/*0132*/	int32 unknown0132;
};

/*
 * OP_Discipline from client size = 4
 */
struct ClientDiscipline_Struct {
    uint8	disc_id;	// There are only a few discs < 60
    uint8	unknown3[3];	// Which leaves room for ??
};

/*
** Memorize Spell Struct
** Length: 12 Bytes
**
*/
struct MemorizeSpell_Struct { 
int32 slot;     // Spot in the spell book/memorized slot 
int32 spell_id; // Spell id (200 or c8 is minor healing, etc) 
int32 scribing; // 1 if memorizing a spell, set to 0 if scribing to book 
		// 2 if un-memming
}; 

/*
** Spell Cast On Struct
** Length: 36 Bytes
**
*/
struct CastOn_Struct {

/*00*/	int16	target_id;
/*02*/	int16	source_id;
/*04*/	int8	source_level;
/*05*/	int8	unknown1;
/*06*/	int8	unknown2; // = 0A
/*07*/	int8	unknown_zero1[7];
/*16*/	float	heading;
/*20*/	int8	unknown_zero2[4];
/*22*/	int32	action;
/*26*/	int8	unknown3;
/*27*/	int16	spell_id;
/*29*/	int8	unknown29;
/*30*/	int8	unknown30; //always 04??
};
struct CastBuff_Struct{
/*00*/	int16	target_id;
/*02*/	int16	source_id;
/*04*/	int8	unknown4; //always E7??
/*05*/	int16	spell_id; //always E7??
/*07*/	int8	unknown_zero7[8];
/*17*/	float	heading2;
/*21*/	int8	unknown_zero21[4];
};




/*
** Make Charmed Pet
** Length: 6 Bytes
**
*/
struct Charm_Struct {
/*00*/	int16	owner_id;
/*02*/	int16	pet_id;
/*04*/	int16	command;    // 1: make pet, 0: release pet
};

struct InterruptCast_Struct
{
	int32 spawnid; //message id
	int32 messageid;
	//char	message[0];
};

struct DeleteSpell_Struct
{
/*000*/sint16	spell_slot;
/*002*/int8	unknowndss002[2];
/*004*/int8	success;
/*005*/int8	unknowndss006[3];
/*008*/
};

struct ManaChange_Struct
{                  
	int32	new_mana;                  // New Mana AMount
	int32	stamina;                  // New Mana AMount
	int32	spell_id;                  // New Mana AMount

};

struct SwapSpell_Struct 
{ 
	int32 from_slot; 
	int32 to_slot; 


}; 

struct BeginCast_Struct
{
	// len = 8
/*000*/	int16	caster_id;

/*002*/	int16	spell_id;
/*004*/	int32	cast_time;		// in miliseconds
};

struct _Buff_Struct
{
/*000*/uint32 target_id;
/*004*/uint32 b_unknown1;
/*008*/uint16 spell_id;
/*010*/uint32 b_unknown2;
/*014*/uint16 b_unknown3;
/*016*/uint32 buff_slot;
/*020*/
};

struct CastSpell_Struct
{
	int32	slot;
	int32	spell_id;
	int32	inventoryslot;  // slot for clicky item, 0xFFFF = normal cast
	int32	target_id;
	//int32	cs_unknown2;
	int8    cs_unknown[4];
};

#define AT_Level		1	// the level that shows up on /who
/*2 = No Effect?*/
#define AT_Invis		3
#define AT_PVP			4
#define AT_Light		5
/*6 = No Effect?*/
#define AT_Position		14	// 100=standing, 110=sitting, 111=ducking, 115=feigned, 105=looting
#define AT_SpawnID		15	// i guess, kinda looks like that's what it's doing
#define AT_LD			18
#define AT_HP			17	// Client->Server, my HP has changed (like regen tic)
#define AT_Levitate		19	// 0=off, 1=flymode, 2=levitate
#define AT_GuildID		22
#define AT_GuildRank	23	// 0=member, 1=officer, 2=leader
#define AT_AFK			24
#define AT_Size			29	// spawn's size
#define AT_NPCName		30	// change PC's name's color to NPC color (unreversable)
#define AT_Trader		31  // Bazzar Trader Mode

/*
** SpawnAppearance_Struct
** Changes client appearance for all other clients in zone
** Size: 8 bytes
** Used in: OP_SpawnAppearance
**
*/
struct SpawnAppearance_Struct
{
/*0000*/ uint16 spawn_id;          // ID of the spawn
/*0002*/ uint16 type;              // Values associated with the type
/*0004*/ uint32 parameter;         // Type of data sent
/*0008*/
};


#if 1
struct SpellBuff_Struct
{
/*000*/	int8	slotid;
/*001*/ int8	level;
/*002*/	int16  effect;				// ***Placeholder
/*004*/	int32	spellid;
/*008*/ int32	duration;
/*012*/	int8	Unknown012[4];
};
#else
struct SpellBuff_Struct
{
/*000*/	int8  b_unknown1[5];		// ***Placeholder
/*005*/	int8  level;				// Level of person who casted buff
/*006*/	int16  effect;				// ***Placeholder
/*008*/	int32 spellid;				// Spell
/*0012*/	int32 duration;				// Duration in ticks
};
#endif

// Length: 24
struct SpellBuffFade_Struct {
/*000*/	uint32 entityid;
/*004*/	int8 slot;
/*005*/	int8 level;
/*006*/	int8 effect;
/*007*/	int8 unknown7;
/*008*/	uint32 spellid;
/*012*/	uint32 unknown012;
/*016*/	uint32 unknown016;
/*020*/	uint32 slotid;
/*024*/	uint32 bufffade;
/*028*/
};

// Length: 10
struct ItemProperties_Struct {

int8	unknown01[2];
int8	charges;
int8	unknown02[13];
};

struct GMTrainee_Struct {
	/*000*/ uint32 npcid;
	/*004*/ uint32 playerid;
	/*008*/ uint32 skills[76];
	/*304*/ int8 unknown000[144];
};

struct GMTrainEnd_Struct {
 // solar: updated 9/01/03
	/*000*/ int16 npcid;
	/*002*/ int16 unknown02;
	/*004*/ int16 spawn_id;
	/*006*/ int16 unknown06;
	/*008*/
};

struct GMSkillChange_Struct {
/*000*/	int16		npcid;
/*002*/ int8		unknown1[2];    // something like PC_ID, but not really. stays the same thru the session though
/*002*/ int16       skillbank;      // 0 if normal skills, 1 if languages
/*002*/ int8		unknown2[2];
/*008*/ uint16		skill_id;
/*010*/ int8		unknown3[2];
};

// all strings zero terminated
#if 0
struct CommonMessage_Struct {
    uint16 unknown[3]; // 00 00 2A 02 0A 00
    char rest[0]; // npcname[at least 16], code[5], pcname[at least 15]
};
#else
struct CommonMessage_Struct {
    uint32 unknown[3]; // 0000 0000 2A02 0000 0A00 0000
    char rest[0]; // npcname[at least 16], code[5], pcname[at least 15]
};
#endif

struct OpCode4141_Struct {
    uint16 unknown;
};

/* 
** Diety List
*/
#define DEITY_UNKNOWN			0
#define DEITY_AGNOSTIC			396
#define DEITY_BRELL				202
#define DEITY_CAZIC				203
#define DEITY_EROL				204
#define DEITY_BRISTLE			205
#define DEITY_INNY				206
#define DEITY_KARANA			207
#define DEITY_MITH				208
#define DEITY_PREXUS			209
#define DEITY_QUELLIOUS			210
#define DEITY_RALLOS			211
#define DEITY_SOLUSEK			213
#define DEITY_TRIBUNAL			214
#define DEITY_TUNARE			215

//Guessed:
#define DEITY_BERT				201	
#define DEITY_RODCET			212
#define DEITY_VEESHAN			216

/*
** Name Generator Struct
** Length: 72 bytes
** OpCode: 0x0290
*/
struct NameGeneration_Struct
{
/*0000*/	int32	race;
/*0004*/	int32	gender;
/*0008*/	char	name[64];
/*0072*/
};

/*
** Character Creation struct
** Length: 140 Bytes
** OpCode: 0x0113
*/
struct CharCreate_Struct
{
	/*0000*/	int32	class_; //guess
	/*0004*/	char	name[64];
	/*0068*/	int32	haircolor; //guess
	/*0072*/	int32	eyecolor1; //guess
	/*0076*/	int32	eyecolor2; //guess
	/*0080*/	sint32	gender;
	/*0084*/	sint32	race;
	/*0088*/	sint32	start_zone;
	/*0092*/	sint32	hairstyle;
	/*0096*/	int32	deity;
	///*0072*/	sint32	deity;
	
	
	// 0 = odus
	// 1 = qeynos
	// 2 = halas
	// 3 = rivervale
	// 4 = freeport
	// 5 = neriak
	// 6 = gukta/grobb
	// 7 = ogguk
	// 8 = kaladim
	// 9 = gfay
	// 10 = felwithe
	// 11 = akanon
	// 12 = cabalis
	// 13 = shar vahl


/*0100*/	sint32	STR;
/*0104*/	sint32	STA;
/*0108*/	sint32	AGI;
/*0112*/	sint32	DEX;
/*0116*/	sint32	WIS;
/*0120*/	sint32	INT;
/*0124*/	sint32	CHA;
/*0128*/	int32	beard;//guess
/*0132*/	int32	beardcolor;//guess
/*0136*/	int32	face;
/*0140*/
};

/*
 *Used in PlayerProfile
 */
struct AA_Array
{
	uint8 AA;
	int8 value;	
};


/*
** Player Profile
**
** Length: 4308 bytes
** OpCode: 0x006a
 */
#define MAX_PP_LANGUAGE		28
#define MAX_PP_SPELLBOOK	400
#define MAX_PP_MEMSPELL		8
#define MAX_PP_SKILL		75
struct PlayerProfile_Struct
{
/*0000*/	uint32				checksum;			// Checksum from CRC32::SetEQChecksum
/*0004*/	char				name[64];			// Name of player sizes not right
/*0068*/	char				last_name[32];		// Last name of player sizes not right
/*0100*/	uint32				gender;				// Player Gender - 0 Male, 1 Female
/*0104*/	uint32				race;				// Player race
/*0108*/	uint32				class_;				// Player class
/*0112*/	uint32				unknown0112;		//
/*0116*/	uint32				level;				// Level of player (might be one byte)
/*0120*/	uint32				bind_zone_id;		// Zone player is bound in
/*0124*/	float				bind_x;				// Bind loc x coord
/*0128*/	float				bind_y;				// Bind loc y coord
/*0132*/	float				bind_z;				// Bind loc z coord
/*0136*/	uint8				unknown0136[4];		//
/*0140*/	uint32				deity;				// deity
/*0144*/	uint32				expAA;				// all FF initially
/*0148*/	uint32				birthday;			// characters bday
/*0152*/	uint32				lastlogin;			// last login or zone time
/*0156*/	uint32				timeplayed;			// in minutes
/*0160*/	uint8				pvp;			// Sta bar % depleted (ie, 30 = 70% sta)
/*0161*/	uint8				fatigue;				// 1=pvp, 0=not pvp
/*0162*/	uint8				unknown0162;		// Level again.

/*0163*/	uint8				anon;				// 2=roleplay, 1=anon, 0=not anon
/*0164*/	uint8				gm;					// 1=gm, 0=not gm
/*0165*/	uint8				unknown0165[43];	//
/*0208*/	uint8				ability_time_seconds;
/*0209*/	uint8				ability_number; //ability used
/*0210*/	uint8				ability_time_minutes;
/*0211*/	uint8				unknown211;//place holder
/*0212*/	uint8				haircolor;			// Player hair color
/*0213*/	uint8				beardcolor;			// Player beard color
/*0214*/	uint8				eyecolor1;			// Player left eye color
/*0215*/	uint8				eyecolor2;			// Player right eye color
/*0216*/	uint8				hairstyle;			// Player hair style
/*0217*/	uint8				beard;				// Beard type

/*0218*/	uint8				unknown0218[2];		// @bp Spacer/Flag?

/*0220*/	uint32				item_material[9];	// Item texture/material of worn/held items
/*0256*/	uint8				unknown0256[52];
/*0308*/	Color_Struct		item_tint[9];
/*0344*/	AA_Array			aa_array[122];		// Length may not be right but i assume no class has more than this

/*0588*/	char				servername[64];		// length probably not right
/*0652*/	uint32				guildid;		//
/*0656*/	uint32				exp;				// Current Experience
/*0660*/	uint32				points;				// Unspent Practice points
/*0664*/	uint32				mana;				// current mana
/*0668*/	uint32				cur_hp;				// current hp
/*0672*/	uint32				unknown0672;		// 0x05
/*0676*/	uint32				STR;				// Strength
/*0680*/	uint32				STA;				// Stamina
/*0684*/	uint32				CHA;				// Charisma
/*0688*/	uint32				DEX;				// Dexterity
/*0692*/	uint32				INT;				// Intelligence
/*0696*/	uint32				AGI;				// Agility
/*0700*/	uint32				WIS;				// Wisdom
/*0704*/	uint8				face;				// Player face
/*0705*/	uint8				unknown0705[11];	// ?
/*0716*/	int32				unknown712[9];		// RR GG BB 00
/*0752*/	uint8				languages[MAX_PP_LANGUAGE];
/*0780*/	uint8				unknown0780[4];		// All 0x00
/*0784*/	int32				spell_book[MAX_PP_SPELLBOOK];
/*2384*/	uint8				unknown2384[448];	// all 0xff   
/*2832*/	int32				mem_spells[MAX_PP_MEMSPELL];
/*2864*/	uint8				unknown2864[32];	//
/*2896*/	uint32				unknown2896;		//
/*2900*/	float				y;					// Player y position
/*2904*/	float				x;					// Player x position
/*2908*/	float				z;					// Player z position
/*2912*/	float				heading;			// Direction player is facing
/*2916*/	uint8				unknown2909[4];		//
/*2920*/	uint32				platinum;			// Platinum Pieces on player
/*2922*/	uint32				gold;				// Gold Pieces on player
/*2928*/	uint32				silver;				// Silver Pieces on player
/*2932*/	uint32				copper;				// Copper Pieces on player
/*2936*/	uint32				platinum_bank;		// Platinum Pieces in Bank
/*2940*/	uint32				gold_bank;			// Gold Pieces in Bank
/*2944*/	uint32				silver_bank;		// Silver Pieces in Bank
/*2948*/	uint32				copper_bank;		// Copper Pieces in Bank
/*2952*/	uint32				platinum_cursor;	// Platinum on cursor
/*2956*/	uint32				gold_cursor;		// Gold on cursor
/*2960*/	uint32				silver_cursor;		// Silver on cursor
/*2964*/	uint32				copper_cursor;		// Copper on cursor
/*2968*/	uint32				platinum_shared;	// Platinum shared between characters
/*2972*/	uint8				unknown2972[16];   	// @bp unknown skills?
/*2992*/	uint32				skills[MAX_PP_SKILL];
/*3288*/	uint32				unknown_skills[46];	// @bp unknown skills?
/*3472*/	uint8				unknown3472[88];	//
/*3560*/	uint32				perAA;				// % on the AA exp bar
/*3564*/	uint8				unknown3564[12];	//28
/*3576*/	uint8				pvp2;	//
/*3591*/	uint8				unknown3591[15];	//
/*3592*/	uint32				zone_change_count;	// Number of times user has zoned in their career (guessing)
/*3596*/	uint8				unknown3596[20];	//
/*3616*/	int32				ability_down;
/*3620*/	uint8				unknown3620[28];	//
/*3648*/	int32				expansion;		// expansion setting
/*3652*/	sint32				unknown3648;
/*3656*/	char				unknown3656[16];	//
/*3672*/	sint32				hunger_level;
/*3676*/	sint32				thirst_level;
/*3680*/	int32				ability_up;
/*3684*/	int32				aapoints;		// Wrong but needs to be put somewhere =P
/*3688*/	char				unknown3688[12];				
/*3700*/	uint32				zone_id;			// Current zone of the player
/*3704*/	SpellBuff_Struct		buffs[15];			// Buffs currently on the player
/*3944*/	char 				groupMembers[6][64];		//
/*4328*/	uint8				unknown4328[4]; 		//
/*4332*/	uint32				ldon_guk_points;		// Earned Deepest Guk points - Doodman: guessing based on others that were known
/*4336*/	uint32				ldon_mirugal_points;		// Earned Mirugal' Mebagerie points - Verified by Doodman
/*4340*/	uint32				ldon_mistmoore_points;		// Earned Mismoore Catacombs Points - Verified by Doodman
/*4344*/	uint32				ldon_rujarkian_points;		// Earned Rujarkian Hills points - Verified by Doodman
/*4348*/	uint32				ldon_takish_points;		// Earned Takish points - Doodman: guessing based on others that were known
/*4352*/	uint8				unknown4352[24]; 		//
/*4376*/	uint32				ldon_available_points;		// Earned Rujarkian Hills points - Verified
/*4380*/	uint32				unknown4380[20]; //added in last patch
/*4460*/	uint32				unknown4460; //0xFF FF FF FF
/*4464*/	uint32				unknown4384[110];
/*4904*/	uint32				unknown4644[4];
};

/*
** Client Target Struct
** Length: 2 Bytes
** OpCode: 6221
*/
struct ClientTarget_Struct {
/*000*/	int32	new_target;			// Target ID
};

/*
** Target Rejection Struct
** Length: 12 Bytes
** OpCode: OP_TargetReject
*/
struct TargetReject_Struct {
/*00*/	uint8	unknown00[12];
};

struct PetCommand_Struct {
/*000*/ int8	command;
/*000*/ int8	unknownpcs000[7];
};

/*
** Generic Spawn Struct
** Length: 257 Bytes
** Fields from old struct not yet found:
**	float	size;
**	float	walkspeed;	// probably one of the ff 33 33 33 3f
**	float	runspeed;	// probably one of the ff 33 33 33 3f
**	int8	traptype;	// 65 is disarmable trap, 66 and 67 are invis triggers/traps
**	int8	npc_armor_graphic;	// 0xFF=Player, 0=none, 1=leather, 2=chain, 3=steelplate
**	int8	npc_helm_graphic;	// 0xFF=Player, 0=none, 1=leather, 2=chain, 3=steelplate
**
*/
struct Spawn_Struct
{
/*000*/	int8	npc;	// 0=player,1=npc,2=pc corpse,3=npc corpse,4=???,5=unknown spawn,10=self
/*001*/	int8	eyecolor1;			// Player left eye color
/*002*/	int8	eyecolor2;			// Player right eye color
/*003*/	int8	aa_title; // 0=none, 1=general, 2=archtype, 3=class
/*004*/	int32	dye_rgb[7]; 			// armor dye colors
/*032*/ int8	unknown001[11]; 
/*043*/ int8	class_; //class
/*044*/ int8	unknown044[2]; // *** Placeholder 
/*046*/ int8	cur_hp; //current hp
/*047*/	int8	afk; // 0=not afk, 1=afk
union {
/*048*/	int8	equip_chest2;// Second place in packet for chest texture (usually 0xFF in live packets)
// Not sure why there are 2 of them, but it effects chest texture!
/*048*/	int8	mount_color;// drogmor: 0=white, 1=black, 2=green, 3=red
// horse: 0=brown, 1=white, 2=black, 3=tan
};
/*049*/ int32	race; // race 
/*053*/ int8	beard; // Player beard style
/*054*/ char	name[64]; // name 
/*118*/ int8	haircolor; // hair color 
/*119*/ int8	light; // 0=visible,1=invisible 
/*120*/ int8	invis; 
/*121*/ int8	max_hp; // max hp 
/*122*/ int8	pvp; // 0=Not pvp,1=pvp 
/*123*/ int8	level; 
/*124*/ int8	lfg; // 0=Not lfg,1=lfg 
/*125*/ int32	heading:12; // spawn heading 
	    int32	delta_heading:10; // change in heading 
		int32	animation:10; // animation id 
/*129*/ sint32	deltaX:13; 
		sint32	x:19; 
/*133*/ sint32	y:19; 
		sint32	deltaZ:13; 
/*137*/ sint32	deltaY:13; 
		sint32	z:19; 
/*141*/ int8	beardcolor;
/*142*/ int8	hairstyle;
/*143*/ int8	face;
/*144*/ int8	unknown141[7]; 
/*151*/ float	size; // Size 
/*155*/ int8	helm; 
/*156*/ float	runspeed; // 
/*160*/ int8	gm; // 0=not GM,1=GM 
/*161*/ float	walkspeed; // 
/*165*/ int32	guild_id; // GuildID 
/*169*/ int8	anon; // 0=normal,1=anon,2=roleplaying 
/*170*/ int8	gender; // 0=male,1=female,2=other 
/*171*/ int16	spawn_id; // Id of spawn 
/*173*/ int8	unknown173[3]; 
/*176*/ char	last_name[32]; // lastname 
/*208*/ int32	equipment[9]; 
/*244*/ int8	linkdead; // 0=Not LD, 1=LD 
/*245*/ uint32	bodytype; // Bodytype 
/*249*/	int8	guild_rank;
/*250*/ int8	unknown249[4]; 
/*254*/ uint32	pet_owner_id; 
/*258*/ int16	deity; 
/*260*/ int8	unknown260[7]; 
};

/*
** New Spawn
** Length: 176 Bytes
** OpCode: 4921
*/
struct NewSpawn_Struct
{
	struct Spawn_Struct spawn;	// Spawn Information
};

/*
** Delete Spawn
** Length: 4 Bytes
** OpCode: OP_DeleteSpawn
*/
struct DeleteSpawn_Struct
{
/*00*/ int32 spawn_id;             // Spawn ID to delete
/*04*/
};

/*
** Channel Message received or sent
** Length: 144 Bytes + Variable Length + 1
** OpCode: OP_ChannelMessage
**
*/
struct ChannelMessage_Struct
{
	union {
/*000*/	char	targetname[64];		// Tell recipient
		uint32	type;
	};
/*064*/	char	sender[64];			// The senders name (len might be wrong)
/*128*/	int32	language;			// Language
/*132*/	int32	chan_num;			// Channel
/*136*/	int32	cm_unknown4;		// ***Placeholder
/*140*/	int32	skill_in_language;	// The players skill in this language? might be wrong
/*144*/	char	message[0];			// Variable length message
};

/*
** Special Message
** Length: 4 Bytes + Variable Text Length + 1
** OpCode: OP_SpecialMesg
**
*/
struct SpecialMesg_Struct
{
/*00*/	char	header[3];				// 04 04 00 <-- for #emote style msg
/*03*/	uint32	msg_type;				// Color of text (see MT_*** below)
/*07*/	uint32	target_spawn_id;		// Who is it being said to?
/*11*/	char	message[0];				// What is being said?
};

// msg_type's for custom usercolors 
#define MT_Say					256
#define MT_Tell					257
#define MT_Group				258
#define MT_Guild				259
#define MT_OOC					260
#define MT_Auction				261
#define MT_Shout				262
#define MT_Emote				263
#define MT_Spells				264
#define MT_YouHitOther			265
#define MT_OtherHitsYou			266
#define MT_YouMissOther			267
#define MT_OtherMissesYou		268
#define MT_Broadcasts			269
#define MT_Skills				270
#define MT_Disciplines			271
#define MT_CritMelee			301

/*
** When somebody changes what they're wearing
**      or give a pet a weapon (model changes)
** Length: 16 Bytes
** Opcode: 9220
*/
struct WearChange_Struct{
/*000*/ int16 spawn_id;
/*004*/ int16 material;
/*005*/ int8 blue;
/*006*/ int8 green;
/*007*/ int8 red;
/*008*/ int8 unknown2;
/*009*/ int8 wear_slot_id;
};

/*
** Type:   Bind Wound Structure
** Length: 8 Bytes
** OpCode: 9320
*/
struct bindWoundStruct
{
/*002*/	int16	playerid;			// TargetID
/*004*/	int8	unknown0004[2];		// ***Placeholder
/*006*/	sint32	hp;					// Hitpoints -- Guess
};

/*
** Type:   Zone Change Request (before hand)
** Length: 70 Bytes-2 = 68 bytes 
** OpCode: a320
*/

#define ZONE_ERROR_NOMSG 0
#define ZONE_ERROR_NOTREADY -1
#define ZONE_ERROR_VALIDPC -2
#define ZONE_ERROR_STORYZONE -3
#define ZONE_ERROR_NOEXPANSION -6
#define ZONE_ERROR_NOEXPERIENCE -7

struct ZoneChange_Struct {
/*000*/	char	char_name[64];     // Character Name
/*064*/	uint32	zoneID;
/*068*/ int8	unknown0072[4];
/*072*/	sint8	success;		// =0 client->server, =1 server->client, -X=specific error
/*073*/	int8	unknown0073[3]; // =0 ok, =ffffff error
};

struct Animation_Struct {
	int16 spawn_id;
	int8 animation;
	int8 animation_speed;
};

#if 0
struct Action_Struct
{
	// len = 24
	int16 target;
	int16 source;
	int8 type;
	int16 spell;
	sint32 damage;
	int8 unknown1[12];
};
#endif
struct Action_Struct
{
 /* 00 */	int16 target;
 /* 02 */	int16 source;
 /* 04 */	uint16 level; // 28000a00
 /* 06 */	uint16 unknown06;
 /* 08 */	uint32 unknown08;
 /* 12 */	uint16 unknown16;
 /* 14 */	float heading;
 /* 18 */	uint32 unknown18;
 /* 22 */	int8 type;
 /* 23 */	uint32 unknown23;
 /* 27 */	int16 spell;
 /* 29 */	int16 unknown29;
 /* 31 */
};
struct CombatDamage_Struct{
	int16	target;
	int16	source;
	int8	type; //slashing, etc
	int16	spellid;
	int32	damage;
	int8	unknown11;
	int16	unknown12;
	int8	unknown14; //0x3D?
	int16	unknown15[2];
	int32	unknown19;
};
/*
** Consider Struct
** Length: 24 Bytes
** OpCode: 3721
*/
struct Consider_Struct{
/*000*/ uint32	playerid;               // PlayerID
/*004*/ uint32	targetid;               // TargetID
/*008*/ int32	faction;                // Faction
/*0012*/ int32	level;                  // Level
/*016*/ sint32	cur_hp;                  // Current Hitpoints
/*020*/ sint32	max_hp;                  // Maximum Hitpoints
/*024*/ int8 pvpcon;     // Pvp con flag 0/1
/*025*/ int8	unknown3[3];
};

/*
** Spawn Death Blow
** Length: 32 Bytes
** OpCode: 0114
*/
struct Death_Struct
{
/*000*/	int32	spawn_id;
/*004*/	int32	killer_id;
/*008*/	int32	corpseid;
/*012*/	int32	type;	
/*016*/	uint32	spell_id;
/*020*/ int32	bindzoneid;
/*024*/	int32	damage;
/*028*/	int32	unknown028;
};

/*
** Generic Spawn Position Update
** Length: 18 Bytes
**
*/
struct SpawnPositionUpdate_Struct
{
/*0000*/ uint16		spawn_id;
/*0002*/ uint64		y:19, z:19, x:19, u3:7;
/*0010*/ unsigned short	heading:12,unused2:4;
/*0012*/
};

/*
** Spawn position update
** Length: 18 Bytes
**	Struct sent from server->client to update position of
**	another spawn's position update in zone (whether NPC or PC)
**
*/
struct PlayerPositionUpdateServer_Struct
{
/*00*/	uint16	spawn_id;
/*02*/	uint32	heading:12;
/*02*/	sint32	delta_heading:10;
/*02*/	uint32	animation:10;
/*06*/	sint32	delta_y:13;
/*06*/	sint32	y_pos:19;
/*10*/	sint32	x_pos:19;
/*10*/	sint32	delta_z:13;
/*14*/	sint32	delta_x:13;
/*14*/	sint32	z_pos:19;
/*18*/
};

/*
** Player position update
** Length: 30 bytes
**	Struct sent from client->server to update
**	player position on server
**
*/
struct PlayerPositionUpdateClient_Struct
{
/*00*/	uint16	spawn_id;
/*02*/	float	z_pos;
/*06*/	float	y_pos;
/*10*/	float	delta_y;
/*14*/	float	x_pos;
/*18*/	float	delta_x;
/*22*/	float	delta_z;
/*26*/	uint32	heading:12;
/*26*/	sint32	delta_heading:10;
/*26*/	uint32	animation:10;
};

/*
** Spawn Position Update
** Length: 6 Bytes + Number of Updates * 15 Bytes
** OpCode: a120
*/

struct SpawnPositionUpdates_Struct
{
/*0000*/ int32  num_updates;               // Number of SpawnUpdates
/*0004*/ struct SpawnPositionUpdate_Struct // Spawn Position Update
                     spawn_update[0];
};

/*
** Spawn HP Update
** Length: 6 Bytes
** OpCode: OP_HPUpdate
*/
struct SpawnHPUpdate_Struct
{
/*00*/ uint16	cur_hp;               // Id of spawn to update
/*02*/ sint16	spawn_id;                 // Current hp of spawn
/*04*/ sint16	max_hp;                 // Maximum hp of spawn
/*06*/
};
struct SpawnHPUpdate_Struct2
{
/*00*/ int8	hp;
/*01*/ sint16	spawn_id;
};
/*
** Stamina
** Length: 8 Bytes
** OpCode: 5721
*/
struct Stamina_Struct {
/*00*/ int32 food;                     // (low more hungry 127-0)
/*02*/ int32 water;                    // (low more thirsty 127-0)
};

/*
** Level Update
** Length: 14 Bytes
** OpCode: 9821
*/
struct LevelUpdate_Struct
{
/*00*/ uint32 level;                  // New level
/*04*/ uint32 level_old;              // Old level
/*08*/ uint32 exp;                    // Current Experience
};

/*
** Experience Update
** Length: 14 Bytes
** OpCode: 9921
*/
struct ExpUpdate_Struct
{
/*0000*/ uint32 exp;                    // Current experience value
/*0004*/ uint32 aaxp; // @BP ??
};

/*
** Child struct of Item_Struct:
**	Common item data
**
*/
struct ItemCommon_Struct {
/*index*/
/*018*/	sint32	Unknown018;
/*019*/	uint32	Unknown019;
/*020*/	uint32	Unknown020;
/*021*/	bool	Tradeskills;		// Is this a tradeskill item?
/*022*/	sint8	SvCold;				// Save vs Cold
/*023*/	sint8	SvDisease;			// Save vs Disease
/*024*/	sint8	SvPoison;			// Save vs Poison
/*025*/	sint8	SvMagic;			// Save vs Magic
/*026*/	sint8	SvFire;				// Save vs Fire
/*027*/	sint8	STR;				// Strength
/*028*/	sint8	STA;				// Stamina
/*029*/	sint8	AGI;				// Agility
/*030*/	sint8	DEX;				// Dexterity
/*031*/	sint8	CHA;				// Charisma
/*032*/	sint8	INT;				// Intelligence
/*033*/	sint8	WIS;				// Wisdom
/*034*/	sint32	HP;					// HP
/*035*/	sint32	Mana;				// Mana
/*036*/	sint32	AC;					// AC
/*037*/	uint32	Deity;				// Bitmask of Deities that can equip this item
/*038*/	sint32	SkillModValue;		// % Mod to skill specified in SkillModType
/*039*/	uint32	SkillModType;		// Type of skill for SkillModValue to apply to
/*040*/	sint8	BaneDmg;			// Bane Damage
/*041*/	uint32	BaneDmgRace;		// Bane Damage Race
/*042*/	uint32	BaneDmgBody;		// Bane Damage Body
/*043*/	bool	Magic;				// True=Magic Item, False=not
/*044*/	sint32	casttime2;
	union {
/*045*/	uint8	ProcLevel;			// Proc lvl
/*045*/	uint8	Haste;				// Haste %
	};
/*046*/	uint8	RequiredLevel;		// Required Level to use item
/*047*/	uint32	BardSkillType;		// Bard Skill Type
/*048*/	sint32	BardSkillAmt;		// Bard Skill Amount
/*049*/	sint8	Light;				// Light
/*050*/	uint8	Delay;				// Delay * 10
/*051*/	uint8	RecommendedLevel;	// Recommended level to use item
/*052*/	uint8	RecommendedSkill;	// Recommended skill to use item (refers to primary skill of item)
/*053*/	uint8	ElemDmgType;		// Elemental Damage Type (1=magic, 2=fire)
/*054*/	uint8	ElemDmg;			// Elemental Damage
/*055*/	uint8	EffectType;			// Effect Type: 0=combat proc, 1=clicky, 2=worn, 3=expendable charges, 4=must equip clicky, 5=clicky (again?)
/*056*/	uint8	Range;				// Range of item
/*057*/	uint8	Damage;				// Delay between item usage (in 0.1 sec increments)
/*058*/	uint32	Color;				// RR GG BB 00 <-- as it appears in pc
/*059*/	uint32	Classes;			// Bitfield of classes that can equip item (1 << class#)
/*060*/	uint32	Races;				// Bitfield of races that can equip item (1 << race#)
/*061*/	uint32	Unknown061;
/*062*/	sint32	SpellId;			// Spell Id of effect, if item has one
/*063*/	sint8	MaxCharges;			// Maximum charges items can hold: -1 if not a chargeable item
/*064*/	uint8	Skill;				// Skill (1hs, 2hs, 1hb, etc)
/*065*/	uint8	Material;			// Item material type
/*066*/	float	SellRate;			// Sell rate
/*067*/	uint32	Unknown067;
	union {
/*068*/	uint32	Fulfilment;			// Food fulfilment (How long it lasts)
/*068*/	sint16	CastTime;			// Cast Time for clicky effects, in milliseconds
	};
/*069*/	uint32	Unknown069;
/*070*/	uint32	Unknown070;
/*071*/	sint32	FocusId;			// Focus Effect Id
/*072*/	sint8	CombatEffects;		// PoP: Combat Effects +
/*073*/	sint8	Shielding;			// PoP: Shielding %
/*074*/	sint8	StunResist;			// PoP: Stun Resist %
/*075*/	sint8	StrikeThrough;		// PoP: Strike Through %
/*076*/	uint32	Unknown076;
/*077*/	uint32	Unknown077;
/*078*/	sint8	SpellShield;		// PoP: Spell Shield %
/*079*/	sint8	Avoidance;			// PoP: Avoidance +
/*080*/	sint8	Accuracy;			// PoP: Accuracy +
/*081*/	sint32	FactionMod1;		// Faction Mod 1
/*082*/	sint32	FactionMod2;		// Faction Mod 2
/*083*/	sint32	FactionMod3;		// Faction Mod 3
/*084*/	sint32	FactionMod4;		// Faction Mod 4
/*085*/	sint32	FactionAmt1;		// Faction Amt 1
/*086*/	sint32	FactionAmt2;		// Faction Amt 2
/*087*/	sint32	FactionAmt3;		// Faction Amt 3
/*088*/	sint32	FactionAmt4;		// Faction Amt 4
/*089*/	uint32	Unknown089;
/*090*/	char	CharmFile[32];		// ?
/*091*/	uint32	Unknown091;
/*092*/	uint8	AugSlot1Type;		// LDoN: Augment Slot 1 Type
/*093*/	uint8	AugSlot2Type;		// LDoN: Augment Slot 2 Type
/*094*/	uint8	AugSlot3Type;		// LDoN: Augment Slot 3 Type
/*095*/	uint8	AugSlot4Type;		// LDoN: Augment Slot 4 Type
/*096*/	uint8	AugSlot5Type;		// LDoN: Augment Slot 5 Type
/*097*/	uint32	ldonpointtheme;
/*098*/	uint32	ldonpointcost;
/*099*/	uint32	Unknown099;
};

/*
** Child struct of Item_Struct:
**	Book item data
**
*/
struct ItemBook_Struct {
/*index*/
/*104*/	uint8	BookType;			// 0=rolled up note, 1=book
/*105*/	uint32	Unknown105;
/*106*/	char	File[15];			// ?
};

/*
** Child struct of Item_Struct:
**	Container item data
**
*/
struct ItemContainer_Struct { 
/*index*/
/*100*/	uint8	PackType;			// 0:Small Bag, 1:Large Bag, 2:Quiver, 3:Belt Pouch ... there are 50 types
/*101*/	uint8	Slots;				// Number of slots: can only be 2, 4, 6, 8, or 10
/*102*/	uint8	SizeCapacity;		// 0:TINY, 1:SMALL, 2:MEDIUM, 3:LARGE, 4:GIANT 
/*103*/	uint8	WeightReduction;	// 0->100

// @merth: From old struct, we still haven't located these values:
// Open		// ?
// Combine	// Whether combine button exists or not
};

/*
** Item data
** Items are no longer sent as a struct; they are sent as a |-delimited string
** The following is the order in which they appear - some lines have been
** commented out to show the element should go there, but doesn't belong here
**
*/
struct Item_Struct {
	void SetCache() {
		// Caching calculated values
		attribs = ItemAttribNone;
		
		if (strstr(LoreName, "&") != 0)
			attribs = (ItemAttrib)(attribs | ItemAttribSummoned);
		if (strstr(LoreName, "*") != 0)
			attribs = (ItemAttrib)(attribs | ItemAttribLore);
		if (strstr(LoreName, "#") != 0)
			attribs = (ItemAttrib)(attribs | ItemAttribArtifact);
		if (strstr(LoreName, "~") != 0)
			attribs = (ItemAttrib)(attribs | ItemAttribPendingLore);
		if ((Type == ItemTypeCommon) && Common.Magic)
			attribs = (ItemAttrib)(attribs | ItemAttribMagic);
	};
	
	// Non packet based fields
	uint8		minstatus;
	bool		gm;
	ItemAttrib	attribs;
	
	
	// Packet based fields
/*index*/
//001*/	sint8	Charges;			// Instance data (just here as FYI)
/*002*/	uint32	Unknown002;
//003*/	sint16	CurrentEquipSlot	// Instance data (just here as FYI)
//004*/	uint32	BazaarTraderPrice;	// Instance data (just here as FYI)
/*005*/	uint32	Unknown005;
/*006*/	uint8	Type;				// Item Type: 0=common, 1=container, 2=book (quote precedes field - dunno why)
/*007*/	char	Name[64];			// Name
/*008*/	char	LoreName[80];		// Lore Name: *=lore, &=summoned, #=artifact, ~=pending lore
/*009*/	char	IDFile[30];			// Visible model
/*010*/	uint32	ItemNumber;			// Unique ID (also PK for DB)
/*011*/	uint8	Weight;				// Item weight * 10
/*012*/	uint8	NoRent;				// No Rent: 0=norent, 255=not norent
/*013*/	uint8	NoDrop;				// No Drop: 0=nodrop, 255=not nodrop
/*014*/	uint8	Size;				// Size: 0=tiny, 1=small, 2=medium, 3=large, 4=giant
/*015*/	uint32	EquipSlots;			// Bitfield for which slots this item can be used in
/*016*/	uint32	Cost;				// Item cost (?)
/*017*/	uint32	IconNumber;			// Icon Number

			union   {
/*018->100*/	ItemCommon_Struct		Common;
/*101->104*/	ItemContainer_Struct	Container;
/*105->107*/	ItemBook_Struct			Book;
            };

/*107*/	uint32	Unknown107;
/*108*/	uint32	Unknown108;
};

/*
** Item Packet Struct - Works on a variety of opcodes
** Packet Types: See ItemPacketType enum
**
*/
struct ItemPacket_Struct
{
/*00*/	ItemPacketType	PacketType;
/*04*/	char			SerializedItem[0];
/*xx*/
};

struct Consume_Struct
{
/*0000*/ int32 slot;
/*0004*/ int32 auto_consumed; // 0xffffffff when auto eating e7030000 when right click
/*0008*/ int8  c_unknown1[4];
/*0012*/ int8  type; // 0x01=Food 0x02=Water
/*0013*/ int8  unknown13[3];
};


struct MoveItem_Struct
{
/*0000*/ uint32 from_slot;
/*0004*/ uint32 to_slot;
/*0008*/ uint32 number_in_stack;
};

struct MoveCoin_Struct
{
		 uint32 from_slot;
		 uint32 to_slot;
		 uint32 cointype1;
		 uint32 cointype2;
		 uint32	amount;
};
struct TradeCoin_Struct{
	int32	trader;
	int8	slot;
	int16	unknown5;
	int8	unknown7;
	int32	amount;
};
struct TradeMoneyUpdate_Struct{
	int32	trader;
	int32	type;
	int32	amount;
};
/*
** Surname struct
** Size: 100 bytes
*/
struct Surname_Struct 
{ 
/*0000*/	char name[64];
/*0064*/	uint32 unknown0064;
/*0068*/	char lastname[32];
/*0100*/
}; 

struct GuildsListEntry_Struct {
	char name[64];
};

struct GuildsList_Struct {
	int8 head[64]; // First on guild list seems to be empty...
	GuildsListEntry_Struct Guilds[512];
};

//#define CODE_NEW_GUILD                  0x7b21
struct GuildUpdate_Struct {

#if 0
	GuildsListEntry_Struct entry;
#else
	int32	guildID;
	GuildsListEntry_Struct entry;
#endif
};

/*
** Money Loot
** Length: 22 Bytes
** OpCode: 5020
*/
struct moneyOnCorpseStruct {
/*0000*/ uint8	response;		// 0 = someone else is, 1 = OK, 2 = not at this time
/*0001*/ uint8	unknown1;		// = 0x5a
/*0002*/ uint8	unknown2;		// = 0x40
/*0003*/ uint8	unknown3;		// = 0
/*0004*/ uint32	platinum;		// Platinum Pieces
/*0008*/ uint32	gold;			// Gold Pieces

/*0012*/ uint32	silver;			// Silver Pieces
/*0016*/ uint32	copper;			// Copper Pieces
};

//opcode = 0x5220
// size 292


struct LootingItem_Struct {
/*000*/	int32	lootee;
/*002*/	int32	looter;
/*004*/	int16	slot_id;
/*006*/	int8	unknown3[2];
/*008*/	int32	type;
};

// Guild invite, remove
struct GuildCommand_Struct {
#if 0
	char othername[64];
	char myname[64];
	int16 guildeqid;
	int8 unknown[6]; // for guildinvite all 0's, for remove 0=0x56, 2=0x02
#else
	char othername[64];
	char myname[64];
	int16 guildeqid;
	int8 unknown[2]; // for guildinvite all 0's, for remove 0=0x56, 2=0x02
	int32 officer;
#endif
};

// Opcode OP_GMZoneRequest
// Size = 88 bytes
struct GMZoneRequest_Struct {
/*0000*/	char	charname[64];
/*0064*/	int32	zone_id;
/*0068*/	float	x;
/*0072*/	float	y;
/*0076*/	float	z;
/*0080*/	char	unknown0080[4];
/*0084*/	int32	success;		// 0 if command failed, 1 if succeeded?
/*0088*/
//	/*072*/	sint8	success;		// =0 client->server, =1 server->client, -X=specific error
//	/*073*/	int8	unknown0073[3]; // =0 ok, =ffffff error
};

struct GMSummon_Struct {
/*  0*/	char    charname[64];
/* 30*/	char    gmname[64];
/* 60*/ int32	success;
/* 61*/	int32	zoneID;
/*92*/	sint32  y;
/*96*/	sint32  x;
/*100*/ sint32  z;
/*104*/	int32 unknown2; // E0 E0 56 00
};

struct GMGoto_Struct { // x,y is swapped as compared to summon and makes sense as own packet
/*  0*/	char    charname[64];

/* 30*/	char    gmname[64];
/* 60*/ int32	success;
/* 61*/	int32	zoneID;

/*92*/	sint32  y;
/*96*/	sint32  x;
/*100*/ sint32  z;
/*104*/	int32 unknown2; // E0 E0 56 00
};

struct GMLastName_Struct {
	char name[64];
	char gmname[64];
	char lastname[64];
	int16 unknown[4];	// 0x00, 0x00
					    // 0x01, 0x00 = Update the clients
};

//Combat Abilities
struct CombatAbility_Struct {
	int32 m_id;
	int32 m_atk;
	int32 m_type;
};

//Instill Doubt
struct Instill_Doubt_Struct {
	int8 i_id;
	int8 ia_unknown;
	int8 ib_unknown;
	int8 ic_unknown;
	int8 i_atk;

	int8 id_unknown;
	int8 ie_unknown;
	int8 if_unknown;
	int8 i_type;
	int8 ig_unknown;
	int8 ih_unknown;
	int8 ii_unknown;
};

struct GiveItem_Struct {
	uint16 to_entity;
	sint16 to_equipSlot;
	uint16 from_entity;
	sint16 from_equipSlot;
};

struct RandomReq_Struct {
	int32 low;
	int32 high;
};

/* solar: 9/23/03 reply to /random command; struct from Zaphod */
struct RandomReply_Struct {
/* 00 */	int32 low;
/* 04 */	int32 high;
/* 08 */	int32 result;
/* 12 */	char name[64];
/* 76 */
};

struct LFG_Struct {
/*
Wrong size on OP_LFG. Got: 80, Expected: 68
   0: 00 00 00 00 01 00 00 00 - 00 00 00 00 64 00 00 00  | ............d...
  16: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  32: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  48: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  64: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
Wrong size on OP_LFG. Got: 80, Expected: 68
   0: 00 00 00 00 01 00 00 00 - 3F 00 00 00 41 00 00 00  | ........?...A...
  16: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  32: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  48: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  64: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
Wrong size on OP_LFG. Got: 80, Expected: 68
   0: 00 00 00 00 01 00 00 00 - 3F 00 00 00 41 00 00 00  | ........?...A...
  16: 46 72 75 62 20 66 72 75 - 62 20 66 72 75 62 00 00  | Frub frub frub..
  32: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  48: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
  64: 00 00 00 00 00 00 00 00 - 00 00 00 00 00 00 00 00  | ................
*/
/*000*/	uint32 unknown000;
/*004*/	uint32 value; // 0x00 = off 0x01 = on
/*008*/	uint32 unknown008;
/*012*/	uint32 unknown012;
/*016*/	char	name[64];
};

/*
** LFG_Appearance_Struct
** Packet sent to clients to notify when someone in zone toggles LFG flag
** Size: 8 bytes
** Used in: OP_LFGAppearance
**
*/
struct LFG_Appearance_Struct
{
/*0000*/ uint32 spawn_id;		// ID of the client
/*0004*/ uint8 lfg;				// 1=LFG, 0=Not LFG
/*0005*/ char unknown0005[3];	//
/*0008*/
};


// EverQuest Time Information:
// 72 minutes per EQ Day
// 3 minutes per EQ Hour
// 6 seconds per EQ Tick (2 minutes EQ Time)
// 3 seconds per EQ Minute

struct TimeOfDay_Struct {
	int8	hour;
	int8	minute;
	int8	day;
	int8	month;
	int32	year;
};

// Darvik: shopkeeper structs
struct Merchant_Click_Struct {
/*000*/ int32	npcid;			// Merchant NPC's entity id
/*004*/ int32	playerid;
/*008*/ int8	unknown[8]; /*
0 is e7 from 01 to // MAYBE SLOT IN PURCHASE
1 is 03
2 is 00
3 is 00
4 is ??
5 is ??
6 is 00 from a0 to
7 is 00 from 3f to */
/*
0 is F6 to 01
1 is CE CE
4A 4A
00 00
00 E0
00 CB
00 90
00 3F
*/

};			

struct Merchant_Sell_Struct {
/*000*/	int32	npcid;			// Merchant NPC's entity id
/*004*/	int32	playerid;		// Player's entity id
/*008*/	int32	itemslot;
		int32	unknown12;
/*016*/	int8	quantity;		// Already sold
/*017*/ int8	Unknown016[3];
/*020*/ int32	price;
};
struct Merchant_Purchase_Struct {
/*000*/	int32	npcid;			// Merchant NPC's entity id
/*004*/	int32	itemslot;		// Player's entity id
/*008*/	int32	quantity;
/*012*/	int32	price;
};

struct Adventure_Purchase_Struct {
/*000*/	int32	npcid;
/*004*/	int32	itemid;
/*008*/	int32	variable;
};

struct AdventurePoints_Update_Struct {
/*000*/	uint32				ldon_available_points;		// Total available points
/*004*/ uint8				unkown_apu004[20];
/*024*/	uint32				ldon_guk_points;		// Earned Deepest Guk points
/*028*/	uint32				ldon_mirugal_points;		// Earned Mirugal' Mebagerie points
/*032*/	uint32				ldon_mistmoore_points;		// Earned Mismoore Catacombs Points
/*036*/	uint32				ldon_rujarkian_points;		// Earned Rujarkian Hills points
/*040*/	uint32				ldon_takish_points;		// Earned Takish points
/*044*/	uint8				unknown_apu042[216];
};

struct Item_Shop_Struct {
	uint16 merchantid;
	int8 itemtype;
	Item_Struct item;
	int8 iss_unknown001[6];
};

/*
** Illusion_Struct
** Changes client visible features
** Size: 104 bytes
** Used In: OP_Illusion, #face, Mob::SendIllusionPacket()
** Fields from the deprecated struct:
**	int8	unknown_26; //Always 26
**	int8	haircolor;
**	int8	beardcolor;
**	int8	eyecolor1; // the eyecolors always seem to be the same, maybe left and right eye?
**	int8	eyecolor2;
**	int8	hairstyle;
**	int8	aa_title;
**	int8	luclinface; // and beard
**
*/
struct Illusion_Struct {
/*000*/	uint32	spawnid;
/*004*/	uint16	race;
/*006*/	char	unknown006[2];
/*008*/	uint8	gender;
/*009*/	uint8	texture;	
/*010*/	uint8	helmtexture;
/*011*/ uint8	unknown011;
/*012*/ uint32	face;
/*016*/	char	unknown020[88];
/*104*/
};

struct MerchantItemD_Struct {
/*0000*/	int8 unknown0000[4];
/*0004*/	Item_Struct item;
/*0500*/
};

struct ItemOnCorpse_Struct {
/*00*/	int16 count;
/*xx*/	struct MerchantItemD_Struct item[0];
};

struct MerchantItem_Struct {
/*00*/	MerchantItemD_Struct packets[0];
};

struct ZonePoint_Entry {
/*0000*/	int32	iterator;
/*0004*/	float	x;
/*0008*/	float	y;
/*0012*/	float	z;
/*0016*/	float	heading;
/*0020*/	int16	zoneid;
/*0022*/	int16	zoneinstance; // LDoN instance
};

struct ZonePoints {
/*0000*/	int32	count;
/*0004*/	struct	ZonePoint_Entry zpe[0]; // Always add one extra to the end after all zonepoints
};

struct SkillUpdate_Struct {
/*00*/	uint32 skillId;
/*04*/	uint32 value;
/*08*/
};

struct ZoneUnavail_Struct {
	//This actually varies, but...
	char zonename[16];
	short int unknown[4];
};

struct GroupGeneric_Struct {
	char name1[64];
	char name2[64];
};

struct GroupUpdate_Struct {	
/*0000*/	int32	action;
/*0004*/	char	yourname[64];
/*0068*/	char	membername[5][64];
/*0388*/	char	leadersname[64];
/*0452*/	int8	unknown[80];
};
struct GroupJoin_Struct {	
/*0000*/	int32	action;
/*0004*/	char	yourname[64];
/*0068*/	char	membername[64];
/*0132*/	int8	unknown[84];
};
struct FaceChange_Struct {
/*000*/	int8	haircolor;
/*001*/	int8	beardcolor;
/*002*/	int8	eyecolor1; // the eyecolors always seem to be the same, maybe left and right eye?
/*003*/	int8	eyecolor2;
/*004*/	int8	hairstyle;
/*005*/	int8	woad; //Face Overlay? (barbarian only)
/*006*/	int8	face; // and beard
};

/*
** Trade request from one client to another
** Used to initiate a trade
** Size: 8 bytes
** Used in: OP_TradeRequest
*/
struct TradeRequest_Struct {
/*00*/	uint32 to_mob_id;
/*04*/	uint32 from_mob_id;
/*08*/
};

/*
** Item to trade struct
** User is placing an item into the trade window
** Size: 512 bytes
** Used In: OP_ItemToTrade
**
*/
struct ItemToTrade_Struct {
/*0000*/	uint32 player_id;
/*0004*/	int16 to_slot;
/*0006*/	char unknown0006[3];
/*0009*/	Item_Struct item;
/*0505*/	char unknown0505[7];
/*0512*/
};

/*
** Cancel Trade struct
** Sent when a player cancels a trade
** Size: 8 bytes
** Used In: OP_CancelTrade
**
*/
struct CancelTrade_Struct {
/*00*/	int32 fromid;
/*04*/	int32 action;
/*08*/
};

struct PetitionUpdate_Struct {
	int32 petnumber;    // Petition Number
	int32 color;		// 0x00 = green, 0x01 = yellow, 0x02 = red
	int32 status;
	time_t senttime;    // 4 has to be 0x1F
	char accountid[32];
	char gmsenttoo[64];
	long quetotal;
	char charname[64];
};

/*struct Petition_Struct {
	int32 petnumber;
	int32 urgency;
	char accountid[32];
	char lastgm[64];
	int32	zone;
	//char zone[32];
	char charname[32];
	int32 charlevel;
	int32 charclass;
	int32 charrace;
	time_t senttime; // Time?
	int32 checkouts;
	int32 unavail;
	int8 unknown5[4];
	char petitiontext[1024];
};*/

struct Petition_Struct {
	int32 petnumber;
	int32 urgency;
	char accountid[32];
	char lastgm[32];
	int32	zone;
	//char zone[32];
	char charname[64];
	int32 charlevel;
	int32 charclass;
	int32 charrace;
	int32 unknown;
	//time_t senttime; // Time?
	int32 checkouts;
	int32 unavail;
	//int8 unknown5[4];
	time_t senttime;
	int32 unknown2;
	char petitiontext[1024];
	char gmtext[1024];
};


struct Who_All_Struct { // 76 length total
/*000*/	char	whom[64];
/*064*/	int32	wrace;		// FF FF = no race

/*066*/	int32	wclass;		// FF FF = no class
/*068*/	int32	lvllow;		// FF FF = no numbers
/*070*/	int32	lvlhigh;	// FF FF = no numbers
/*072*/	int32	gmlookup;	// FF FF = not doing /who all gm
/*074*/	int32	unknown074;
/*076*/	int8	unknown076[64];
/*140*/
};

struct Stun_Struct { // 4 bytes total 
	int32 duration; // Duration of stun
};

/*
** New Combine Struct
** Client requesting to perform a tradeskill combine
** Size: 4 bytes
** Used In: OP_TradeSkillCombine
** Last Updated: Oct-15-2003
**
*/
struct NewCombine_Struct { 
/*00*/	sint16	container_slot;
/*02*/	char	unknown02[2];
/*04*/
};

// OP_Emote
struct Emote_Struct {
/*0000*/	int32 unknown01;
/*0004*/	char message[1024];
/*1028*/
};

// OP_EmoteAnim
struct EmoteAnim_Struct {
/*00*/	int16 spawnid;
/*02*/	int8 action;
/*03*/	int8 value;
/*04*/
};

// Inspect 
struct Inspect_Struct { 
	int16 TargetID; 
	int16 PlayerID; 
}; 

//OP_SetDataRate
struct SetDataRate_Struct {
	float newdatarate;
};

//OP_SetServerFilter
struct SetServerFilter_Struct {
	/*0000*/ int32	damageshield;
	/*0004*/ int32	npcspells;
	/*0008*/ int32	pcspells;
	/*0012*/ int32	bardsongs;
	/*0016*/ int32	unknown;
	/*0020*/ int32	guildsay;
	/*0024*/ int32	socials;
	/*0028*/ int32	group;
	/*0032*/ int32	shout;
	/*0036*/ int32	auction;
	/*0040*/ int32	ooc;
	/*0044*/ int32	mymisses;
	/*0048*/ int32	othermisses;
	/*0052*/ int32	otherhits;
	/*0056*/ int32	atkmissesme;
	/*0060*/ int32	critspells;
	/*0064*/ int32	critmelee;
	/*0068*/ int32	spelldamage;
	/*0072*/ int32	dotdamage;
	/*0076*/ int32	mypethits;
	/*0080*/ int32	mypetmisses;
};
#define FILTER_DAMAGESHIELD 0
#define FILTER_NPCSPELLS	1
#define FILTER_PCSPELLS		2
#define FILTER_BARDSONGS	3
#define FILTER_GUILDSAY		4
#define FILTER_SOCIALS		5
#define FILTER_GROUP		6
#define FILTER_SHOUT		7
#define FILTER_AUCTION		8
#define FILTER_OOC			9
#define FILTER_MYMISSES		10
#define FILTER_OTHERMISSES	11
#define FILTER_OTHERHITS	12
#define FILTER_ATKMISSESME	13
#define FILTER_CRITSPELLS	14
#define FILTER_CRITMELEE	15
#define FILTER_SPELLDAMAGE	16
#define FILTER_DOTDAMAGE	17
#define FILTER_MYPETHITS	18
#define FILTER_MYPETMISSES	19
//Op_SetServerFilterAck
struct SetServerFilterAck_Struct {
	int8 blank[8];
};
struct IncreaseStat_Struct{
	/*0000*/	int8	unknown0;
	/*0001*/	int8	str;
	/*0002*/	int8	sta;
	/*0003*/	int8	agi;
	/*0004*/	int8	dex;
	/*0005*/	int8	int_;
	/*0006*/	int8	wis;
	/*0007*/	int8	cha;
	/*0008*/	int8	fire;
	/*0009*/	int8	cold;
	/*0010*/	int8	magic;
	/*0011*/	int8	poison;
	/*0012*/	int8	disease;
	/*0013*/	char	unknown13[116];
	/*0129*/	int8	str2;
	/*0130*/	int8	sta2;
	/*0131*/	int8	agi2;
	/*0132*/	int8	dex2;
	/*0133*/	int8	int_2;
	/*0134*/	int8	wis2;
	/*0135*/	int8	cha2;
	/*0136*/	int8	fire2;
	/*0137*/	int8	cold2;
	/*0138*/	int8	magic2;
	/*0139*/	int8	poison2;
	/*0140*/	int8	disease2;
};
#define	STAT_STR		0
#define	STAT_STA		1
#define	STAT_AGI		2
#define	STAT_DEX		3
#define	STAT_INT		4
#define	STAT_WIS		5
#define	STAT_CHA		6
#define	STAT_FIRE		7
#define	STAT_COLD		8
#define	STAT_MAGIC		9
#define	STAT_POISON		10
#define	STAT_DISEASE	11

struct GMName_Struct {
	char oldname[64];
	char gmname[64];
	char newname[64];
	int8 badname;
	int8 unknown[3];
};

struct GMDelCorpse_Struct {
	char corpsename[64];
	char gmname[64];
	int8 unknown;
};

struct GMKick_Struct {
	char name[64];
	char gmname[64];
	int8 unknown;
};


struct GMKill_Struct {
	char name[64];
	char gmname[64];
	int8 unknown;
};


struct GMEmoteZone_Struct {
	char text[512];
};

struct Unknown_Struct {
	int8 unknown1[60*4];
};

// This is where the Text is sent to the client.
// Use ` as a newline character in the text.
// Variable length.
struct BookText_Struct {
	int16 unknown0;
	char* booktext; // Variable Length
};
// This is the request to read a book.
// This is just a "text file" on the server
// or in our case, the 'name' column in our books table.
struct BookRequest_Struct {
	int16 unknown0;
	char txtfile[20];
};

/*
** Object/Ground Spawn struct
** Used for Forges, Ovens, ground spawns, items dropped to ground, etc
** Size: 92 bytes
** OpCodes: OP_CreateObject
** Last Updated: Oct-17-2003
**
*/
struct Object_Struct {
/*00*/	uint32	linked_list_addr[2];// <Zaphod> They are, get this, prev and next, ala linked list
/*08*/	uint16	unknown008[2];		//
/*12*/	uint32	drop_id;			// Unique object id for zone
/*16*/	uint32	zone_id;			// Redudant, but: Zone the object appears in
/*20*/	uint32	unknown020[2];		//
/*28*/	float	heading;			// heading
/*32*/	float	z;					// z coord
/*36*/	float	y;					// y coord
/*40*/	float	x;					// x coord
/*44*/	char	object_name[16];	// Name of object, usually something like IT63_ACTORDEF
/*60*/	uint32	unknown060[5];		//
/*80*/	uint32	object_type;		// Type of object, not directly translated to OP_OpenObject
/*84*/	uint32	unknown084[1];		//
/*88*/	uint32	spawn_id;			// Spawn Id of client interacting with object
/*92*/
};
//<Zaphod> 01 = generic drop, 02 = armor, 19 = weapon
//[13:40] <Zaphod> and 0xff seems to be indicative of the tradeskill/openable items that end up returning the old style item type in the OP_OpenObject

/*
** Click Object Struct
** Client clicking on zone object (forge, groundspawn, etc)
** Size: 8 bytes
** Last Updated: Oct-17-2003
**
*/
struct ClickObject_Struct {
/*00*/	uint32 drop_id;
/*04*/	uint32 player_id;
/*08*/
};

/*
** Click Object Acknowledgement Struct
** Response to client clicking on a World Container (ie, forge)
** Size: 24 bytes
** Last Updated: Oct-17-2003
**
*/
struct ClickObjectAck_Struct {
/*00*/	uint32	player_id;	// Entity Id of player who clicked object
/*04*/	uint32	drop_id;	// Zone-specified unique object identifier
/*08*/	uint32	open;		// 1=opening, 0=closing
/*12*/	uint32	type;		// See object.h, "Object Types"
/*16*/	uint32	unknown16;	//
/*20*/	uint32	icon;		// Icon to display for tradeskill containers
/*24*/
};

/*
** Struct with two uses
**
*/
struct CloseContainer_Struct : public ClickObjectAck_Struct {
};

/*
** Generic Door Struct
** Length: 52 Octets
** Used in: 
**    cDoorSpawnsStruct(f721)
**
*/
struct Door_Struct
{
/*0000*/ char    name[16];            // Filename of Door // Was 10char long before... added the 6 in the next unknown to it: Daeken M. BlackBlade
//		 uint8	 unknown0008[6];	 // This is "supposed" to be with name hehe
/*0016*/ float   yPos;               // y loc
/*0020*/ float   xPos;               // x loc
/*0024*/ float   zPos;               // z loc
/*0028*/ float	 heading;
/*0032*/ uint8   unknown0032[6];
/*0038*/ uint8   unknown0038;
/*0039*/ int8   auto_return;
/*0040*/ uint8   initialstate;
/*0041*/ uint16  unknown041;
/*0043*/ uint8 unknown0043;
/*0044*/ uint8  doorId;             // door's id #
/*0045*/ uint8   opentype;
/*
 *  Open types:
 * 66 = PORT1414 (Qeynos)
 * 55 = BBBOARD (Qeynos)
 * 100 = QEYLAMP (Qeynos)
 * 56 = CHEST1 (Qeynos)
 * 5 = DOOR1 (Qeynos)
 */
/*0046*/ uint8  size;
/*0047*/ uint16  holdstateforever;               // guess..
/*0048*/ uint8  unknown0048[3];     // ***Placeholder
/*0052*/ uint32 unknown0052;
};



struct DoorSpawns_Struct	//SEQ
{
	struct Door_Struct doors[0];
};

/*
 OP Code: 	Op_ClickDoor
 Size:		16

 10/10/2003-Doodman	Filled in struct
*/
struct ClickDoor_Struct {
/*000*/	int8	doorid;
/*001*/	int8	unknown001[3];
/*004*/	int8	picklockskill;
/*005*/	int8	unknown005[3];
/*008*/ int32	item_id;
/*012*/ int16	player_id;
/*014*/ int8	unknown014[2];
};

struct MoveDoor_Struct {
	int8	doorid;
	int8	action;
};


struct BecomeNPC_Struct {
	int32 id;
	long maxlevel;
};

struct Underworld_Struct {
	float speed;
	float y;
	float x;
	float z;
};

struct Resurrect_Struct	//160
{
	int16	unknown_01;            
	char	zone[15];
	int8	unknown_02[19];
	float	y;
	float	x;
	float	z;
	int32	unknown_02_1;
	char	your_name[64];
	int8	unknown_03[6];
	char	rezzer_name[64];
	int8	unknown_04[2];
	int16	spellid;
	char	corpse_name[64];
	int8	unknown_05[4];
	int32	action;
};

// updates client with alternate advancement (non-AA table) information
// Length: 12 Bytes
// OpCode: 0x01f2
struct AltAdvStats_Struct {
/*000*/  uint32 experience;
/*004*/  uint16 unspent;
/*006*/  uint16	unknown006;
/*008*/  int8	percentage;
/*009*/  int8	unknown009[3];
};
struct UseAA_Struct {
	int32 begin;
	int32 ability;
	int32 end;
};
//  Alternate Advancement table.  holds all the skill levels for the AA skills.
//  Length: 256 Bytes
//  OpCode: 1422
struct PlayerAA_Struct {
/*    0 */  uint16 unknown0;
  union {
    uint16 unnamed[17];
    struct {  
/*    1 */  uint16 innate_strength;
/*    2 */  uint16 innate_stamina;
/*    3 */  uint16 innate_agility;
/*    4 */  uint16 innate_dexterity;
/*    5 */  uint16 innate_intelligence;
/*    6 */  uint16 innate_wisdom;
/*    7 */  uint16 innate_charisma;
/*    8 */  uint16 innate_fire_protection;

/*    9 */  uint16 innate_cold_protection;
/*   10 */  uint16 innate_magic_protection;
/*   11 */  uint16 innate_poison_protection;
/*   12 */  uint16 innate_disease_protection;
/*   13 */  uint16 innate_run_speed;
/*   14 */  uint16 innate_regeneration;
/*   15 */  uint16 innate_metabolism;
/*   16 */  uint16 innate_lung_capacity;
/*   17 */  uint16 first_aid;
    } named;
  } general_skills;
  union {
    uint16 unnamed[17];
    struct {
/*   18 */  uint16 healing_adept;
/*   19 */  uint16 healing_gift;
/*   20 */  uint16 spell_casting_mastery;
/*   21 */  uint16 spell_casting_reinforcement;
/*   22 */  uint16 mental_clarity;
/*   23 */  uint16 spell_casting_fury;
/*   24 */  uint16 chanelling_focus;
/*   25 */  uint16 spell_casting_subtlety;
/*   26 */  uint16 spell_casting_expertise;
/*   27 */  uint16 spell_casting_deftness;
/*   28 */  uint16 natural_durability;
/*   29 */  uint16 natural_healing;
/*   30 */  uint16 combat_fury;
/*   31 */  uint16 fear_resistance;
/*   32 */  uint16 finishing_blow;
/*   33 */  uint16 combat_stability;
/*   34 */  uint16 combat_agility;
    } named;
  } archetype_skills;
  union {
    uint16 unnamed[93];
    struct {
/*   35 */  uint16 mass_group_buff; // All group-buff-casting classes(?)
// ===== Cleric =====
/*   36 */  uint16 divine_resurrection;
/*   37 */  uint16 innate_invis_to_undead; // cleric, necromancer
/*   38 */  uint16 celestial_regeneration;
/*   39 */  uint16 bestow_divine_aura;
/*   40 */  uint16 turn_undead;
/*   41 */  uint16 purify_soul;
// ===== Druid =====
/*   42 */  uint16 quick_evacuation; // wizard, druid
/*   43 */  uint16 exodus; // wizard, druid
/*   44 */  uint16 quick_damage; // wizard, druid
/*   45 */  uint16 enhanced_root; // druid
/*   46 */  uint16 dire_charm; // enchanter, druid, necromancer
// ===== Shaman =====
/*   47 */  uint16 cannibalization;
/*   48 */  uint16 quick_buff; // shaman, enchanter
/*   49 */  uint16 alchemy_mastery;
/*   50 */  uint16 rabid_bear;
// ===== Wizard =====
/*   51 */  uint16 mana_burn;
/*   52 */  uint16 improved_familiar;
/*   53 */  uint16 nexus_gate;
// ===== Enchanter =====
/*   54 */  uint16 unknown54;
/*   55 */  uint16 permanent_illusion;
/*   56 */  uint16 jewel_craft_mastery;
/*   57 */  uint16 gather_mana;
// ===== Mage =====
/*   58 */  uint16 mend_companion; // mage, necromancer
/*   59 */  uint16 quick_summoning;
/*   60 */  uint16 frenzied_burnout;
/*   61 */  uint16 elemental_form_fire;
/*   62 */  uint16 elemental_form_water;
/*   63 */  uint16 elemental_form_earth;
/*   64 */  uint16 elemental_form_air;
/*   65 */  uint16 improved_reclaim_energy;
/*   66 */  uint16 turn_summoned;
/*   67 */  uint16 elemental_pact;
// ===== Necromancer =====
/*   68 */  uint16 life_burn;
/*   69 */  uint16 dead_mesmerization;
/*   70 */  uint16 fearstorm;
/*   71 */  uint16 flesh_to_bone;
/*   72 */  uint16 call_to_corpse;
// ===== Paladin =====
/*   73 */  uint16 divine_stun;
/*   74 */  uint16 improved_lay_of_hands;
/*   75 */  uint16 slay_undead;
/*   76 */  uint16 act_of_valor;
/*   77 */  uint16 holy_steed;
/*   78 */  uint16 fearless; // paladin, shadowknight
/*   79 */  uint16 two_hand_bash; // paladin, shadowknight
// ===== Ranger =====
/*   80 */  uint16 innate_camouflage; // ranger, druid
/*   81 */  uint16 ambidexterity; // all "dual-wield" users
/*   82 */  uint16 archery_mastery; // ranger
/*   83 */  uint16 unknown83;
/*   84 */  uint16 endless_quiver; // ranger
// ===== Shadow Knight =====
/*   85 */  uint16 unholy_steed;
/*   86 */  uint16 improved_harm_touch;
/*   87 */  uint16 leech_touch;
/*   88 */  uint16 unknown88;
/*   89 */  uint16 soul_abrasion;
// ===== Bard =====
/*   90 */  uint16 instrument_mastery;
/*   91 */  uint16 unknown91;
/*   92 */  uint16 unknown92;
/*   93 */  uint16 unknown93;
/*   94 */  uint16 jam_fest;
/*   95 */  uint16 unknown95;
/*   96 */  uint16 unknown96;
// ===== Monk =====
/*   97 */  uint16 critical_mend;
/*   98 */  uint16 purify_body;
/*   99 */  uint16 unknown99;
/*  100 */  uint16 rapid_feign;
/*  101 */  uint16 return_kick;
// ===== Rogue =====
/*  102 */  uint16 escape;
/*  103 */  uint16 poison_mastery;
/*  104 */  uint16 double_riposte; // all "riposte" users
/*  105 */  uint16 unknown105;
/*  106 */  uint16 unknown106;
/*  107 */  uint16 purge_poison; // rogue
// ===== Warrior =====
/*  108 */  uint16 flurry;
/*  109 */  uint16 rampage;
/*  110 */  uint16 area_taunt;
/*  111 */  uint16 warcry;
/*  112 */  uint16 bandage_wound;
// ===== (Other) =====
/*  113 */  uint16 spell_casting_reinforcement_mastery; // all "pure" casters
/*  114 */  uint16 unknown114;
/*  115 */  uint16 extended_notes; // bard
/*  116 */  uint16 dragon_punch; // monk
/*  117 */  uint16 strong_root; // wizard
/*  118 */  uint16 singing_mastery; // bard
/*  119 */  uint16 body_and_mind_rejuvenation; // paladin, ranger, bard
/*  120 */  uint16 physical_enhancement; // paladin, ranger, bard
/*  121 */  uint16 adv_trap_negotiation; // rogue, bard
/*  122 */  uint16 acrobatics; // all "safe-fall" users
/*  123 */  uint16 scribble_notes; // bard
/*  124 */  uint16 chaotic_stab; // rogue
/*  125 */  uint16 pet_discipline; // all pet classes except enchanter
/*  126 */  uint16 unknown126;
/*  127 */  uint16 unknown127;
/*  128 */  uint16 unknown128;
    } named;
  } class_skills;
  union {
    uint16 unnamed[18];
    struct {  
/*  129 */  uint16 advanced_innate_strength;
/*  130 */  uint16 advanced_innate_stamina;
/*  131 */  uint16 advanced_innate_agility;
/*  132 */  uint16 advanced_innate_dexterity;
/*  133 */  uint16 advanced_innate_intelligence;
/*  134 */  uint16 advanced_innate_wisdom;
/*  135 */  uint16 advanced_innate_charisma;
/*  136 */  uint16 warding_of_solusek;
/*  137 */  uint16 blessing_of_eci;

/*  138 */  uint16 marrs_protection;
/*  139 */  uint16 shroud_of_the_faceless;
/*  140 */  uint16 bertoxxulous_gift;
/*  141 */  uint16 new_tanaan_crafting_mastery;
/*  142 */  uint16 planar_power;
/*  143 */  uint16 planar_durability;

    } named;
  } pop_advance;
    union {
    uint16 unnamed[14];
    struct {  
/*  144 */  uint16 unknown144;
/*  145 */  uint16 unknown145;
/*  146 */  uint16 unknown146;
/*  147 */  uint16 unknown147;
/*  148 */  uint16 coup_de_grace;
/*  149 */  uint16 fury_of_the_ages;
/*  150 */  uint16 unknown150;
/*  151 */  uint16 lightning_reflexes;
/*  152 */  uint16 innate_defense;
/*  153 */  uint16 unknown153;
/*  154 */  uint16 unknown154;
/*  155 */  uint16 unknown155;
/*  156 */  uint16 unknown156;
/*  157 */  uint16 unknown157;
/*  158 */  uint16 unknown158;
/*  159 */  uint16 unknown159;
/*  160 */  uint16 unknown160;
/*  161 */  uint16 unknown161;
/*  162 */  uint16 unknown162;
/*  163 */  uint16 unknown163;
/*  164 */  uint16 unknown164;

/*  165 */  uint16 unknown165;
/*  166 */  uint16 hasty_exit;
/*  167 */  uint16 unknown167;
/*  168 */  uint16 unknown168;
/*  169 */  uint16 unknown169;
/*  170 */  uint16 unknown170;
/*  171 */  uint16 unknown171;
/*  172 */  uint16 unknown172;
/*  173 */  uint16 unknown173;
/*  174 */  uint16 unknown174;
/*  175 */  uint16 unknown175;
/*  176 */  uint16 unknown176;
/*  177 */  uint16 unknown177;
/*  178 */  uint16 unknown178;
/*  179 */  uint16 unknown179;
/*  180 */  uint16 unknown180;

/*  181 */  uint16 unknown181;
/*  182 */  uint16 unknown182;
/*  183 */  uint16 unknown183;
/*  184 */  uint16 unknown184;
/*  185 */  uint16 unknown185;
/*  186 */  uint16 unknown186;
/*  187 */  uint16 unknown187;
/*  188 */  uint16 unknown188;
/*  189 */  uint16 unknown189;
/*  190 */  uint16 unknown190;
/*  191 */  uint16 unknown191;
/*  192 */  uint16 unknown192;
/*  193 */  uint16 unknown193;
/*  194 */  uint16 unknown194;
/*  195 */  uint16 unknown195;
/*  196 */  uint16 unknown196;

/*  197 */  uint16 unknown197;
/*  198 */  uint16 unknown198;
/*  199 */  uint16 unknown199;
/*  200 */  uint16 unknown200;
/*  201 */  uint16 unknown201;
/*  202 */  uint16 unknown202;
/*  203 */  uint16 unknown203;
/*  204 */  uint16 unknown204;
/*  205 */  uint16 unknown205;
/*  206 */  uint16 unknown206;
/*  207 */  uint16 unknown207;
/*  208 */  uint16 unknown208;
/*  209 */  uint16 unknown209;
/*  210 */  uint16 unknown210;
/*  211 */  uint16 unknown211;
/*  212 */  uint16 unknown212;


/*  213 */  uint16 unknown213;
/*  214 */  uint16 unknown214;
/*  215 */  uint16 unknown215;
/*  216 */  uint16 unknown216;
/*  217 */  uint16 unknown217;
/*  218 */  uint16 unknown218;
/*  219 */  uint16 unknown219;
/*  220 */  uint16 unknown220;
    } named;
  } pop_ability;
};

struct SetRunMode_Struct {
	int8 mode;
	int8 unknown[3];
};

//EnvDamage is EnvDamage2 without a few bytes at the end.

struct EnvDamage2_Struct {
/*0000*/	int32 id;
/*0004*/	int16 unknown4;
/*0006*/	int32 damage;
/*0010*/	int8 unknown10[12];
/*0022*/	int8 dmgtype; //FA = Lava; FC = Falling
/*0023*/	int8 unknown2[4];
/*0027*/	int16 constant; //Always FFFF
/*0029*/	int16 unknown29;
};

//Bazaar Stuff =D

struct BazaarWindowStart_Struct {
	int8   action;
	int8   unknown1;
	int16  unknown2;
};


struct BazaarWelcome_Struct {
	BazaarWindowStart_Struct beginning;
	int32  traders;
	int32  items;
	int8   unknown1[8];
};

struct BazaarSearch_Struct {
	BazaarWindowStart_Struct beginning;
	int32	traderid;
	int32  class_;
	int32  race;
	int32  stat;
	int32  slot;
	int32  type;
	char   name[64];
	int32	minprice;
	int32	maxprice;
};
struct BazaarInspect_Struct{
	int32 item_id;
	int32 unknown;
	char name[64];
};
struct BazaarReturnDone_Struct{
	int32 type;
	int32 traderid;
	int32 unknown8;
	int32 unknown12;
	int32 unknown16;
};
struct BazaarSearchResults_Struct {
	BazaarWindowStart_Struct beginning;
	int32	numitems;
	int32	item_id;
	int32	seller_nr;
	int32	cost;
	int32	unknown20;
	char	name[64];
};
struct BazaarInspectItem_Struct {
	int8   action;
	int8   unknown1;
#define SIZEOFITEMSTRUCT sizeof(Item_Struct)
	int8   itemdata[SIZEOFITEMSTRUCT];
};

struct SpecialMsg_Struct {
	int8 unknown[6]; //Always: 00 00 37 04 0a 00 for syswide
	char message[];
};

struct ServerSideFilters_Struct {
int8	clientattackfilters; // 0) No, 1) All (players) but self, 2) All (players) but group
int8	npcattackfilters;	 // 0) No, 1) Ignore NPC misses (all), 2) Ignore NPC Misses + Attacks (all but self), 3) Ignores NPC Misses + Attacks (all but group)
int8	clientcastfilters;	 // 0) No, 1) Ignore PC Casts (all), 2) Ignore PC Casts (not directed towards self)
int8	npccastfilters;		 // 0) No, 1) Ignore NPC Casts (all), 2) Ignore NPC Casts (not directed towards self)
};

/*
** Client requesting item statistics
** Size: 32 bytes
** Used In: OP_ItemLinkClick
** Last Updated: Sept-19-2003
**
*/
struct	ItemViewRequest_Struct {
/*000*/	uint32	item_id;
/*004*/	char	unknown004[28];
/*076*/
};

struct ItemViewResponse_Struct {
struct Item_Struct item;
};

/*
 *  Client to server packet
 */
struct PickPocket_Struct {
// Size 18
    uint32 to;
    uint32 from;
    uint8 myskill;
    uint8 unknown0;
    uint8 type; // -1 you are being picked, 0 failed , 1 = plat, 2 = gold, 3 = silver, 4 = copper, 5 = item
    uint8 unknown1; // 0 for response, unknown for input
    uint32 coin;
    uint8 lastsix[2];
};
/*
 * Server to client packet
 */

struct sPickPocket_Struct {
	// Size 28 = coin/fail
	uint32 to;
	uint32 from;
	uint32 myskill;
	uint32 type;
	uint32 coin;
	uint32 unknowns[2];
};
struct sItem_PickPocket_Struct {
	// Size 524 = Item
	uint32 to;
	uint32 from;
	uint32 myskill;
	uint32 type;
	uint32 coin;
	Item_Struct item;
	uint32 unknowns[2];
};

struct BindWound_Struct {
// Size 4
    uint16  to; // entity id
    uint8    type; // This could be an int16
    // 0 or 1 complete, 2 Unknown, 3 ACK, 4 Died, 5 Left, 6 they moved, 7 you moved
    uint8    placeholder; //
};

struct LogServer_Struct {
// Size 252
// Size 252
// Op_Code OP_LOGSERVER
    uint8   unknown252[256];
};

struct ApproveWorld_Struct {
// Size 544
// Op_Code OP_ApproveWorld
    uint8 unknown544[544];
};

struct ClientError_Struct
{
/*00000*/	char	unknown0000[69];
/*00069*/	char	character_name[64];
/*00133*/	char	message[32067];
/*32136*/
};

struct MobHealth
{
	/*0000*/	int8	hp; //health percent
	/*0001*/	int16	id;//mobs id
};

struct ItemLink_Struct {
/*0000*/	int8	unknown[12];
/*0012*/	int8	begin; //always 0x12
/*0013*/	char*	itemnumber;
/*0020*/	int16	unknown20[4];
/*0028*/	char *	itemname;
/*0029*/	int8	end; //always 0x12
};

struct Track_Struct {
	int16 entityid;
	int16 y;
	int16 x;
	int16 z;
};

struct Tracking_Struct {
	Track_Struct Entrys[0];
};

























// @Bigpull: 0.4.4 item struct for InjectToRaw()
struct OLDItem_Struct
{
/*0000*/ char      name[64];        // Name of item
/*0064*/ char      lore[80];        // Lore text
/*0144*/ char      idfile[30];      // This is the filename of the item graphic when held/worn.
									// Changed to 30 bytes for correct item conversion
/*0174*/ uint8	   weight;          // Weight of item
/*0175*/ sint8     nosave;          // Nosave flag 1=normal, 0=nosave, -1=spell?
/*0176*/ sint8     nodrop;          // Nodrop flag 1=normal, 0=nodrop, -1=??
/*0177*/ uint8     size;            // Size of item
/*0178*/ int8      type;
/*0179*/ uint8     unknown0178;     // ***Placeholder
/*0180*/ uint16    item_nr;         // Unique Item number
/*0182*/ uint16    icon_nr;         // Icon Number
/*0184*/ sint16    equipSlot;       // Current Equip slot
/*0186*/ uint8     unknwn0186[2];   // Equip slot cont.?
/*0188*/ uint32    equipableSlots;  // Slots where this item goes
/*0192*/ sint32    cost;            // Item cost in copper
/*0196*/ uint8     unknown0196[32]; // ***Placeholder
union
{
	struct
	{
	// 0228- have different meanings depending on flags
	/*0228*/ sint8    STR;              // Strength
	/*0229*/ sint8    STA;              // Stamina
	/*0230*/ sint8    CHA;              // Charisma
	/*0231*/ sint8    DEX;              // Dexterity
	/*0232*/ sint8    INT;              // Intelligence
	/*0233*/ sint8    AGI;              // Agility
	/*0234*/ sint8    WIS;              // Wisdom
	/*0235*/ sint8    MR;               // Magic Resistance
	/*0236*/ sint8    FR;               // Fire Resistance
	/*0237*/ sint8    CR;               // Cold Resistance
	/*0238*/ sint8    DR;               // Disease Resistance
	/*0239*/ sint8    PR;               // Poison Resistance
	/*0240*/ sint16   HP;               // Hitpoints
	/*0242*/ sint16   MANA;             // Mana
	/*0244*/ sint16   AC;               // Armor Class
	/*0246*/ uint8    MaxCharges;       // Maximum number of charges, for rechargable? (Sept 25, 2002)
	/*0247*/ sint8    GMFlag;           // GM flag 0  - normal item, -1 - gm item (Sept 25, 2002)
	/*0248*/ uint8    light;            // Light effect of this item
	/*0249*/ uint8    delay;            // Weapon Delay
	/*0250*/ uint8    damage;           // Weapon Damage
	/*0251*/ sint8    effecttype0;      // 0=combat, 1=click anywhere w/o class check, 2=latent/worn, 3=click anywhere EXPENDABLE, 4=click worn, 5=click anywhere w/ class check, -1=no effect
	/*0252*/ uint8    range;            // Range of weapon
	/*0253*/ uint8    skill;            // Skill of this weapon, refer to weaponskill chart
	/*0254*/ sint8    magic;            // Magic flag
                        //   00  (0000)  =   ???
                        //   01  (0001)  =  magic
                        //   12  (1100)  =   ???
                        //   14  (1110)  =   ???
                        //   15  (1111)  =   ???
	/*0255*/ sint8    level0;           // Casting level
	/*0256*/ uint8    material;         // Material?
	/*0257*/ uint8    unknown0258[3];   // ***Placeholder
	/*0260*/ uint32   color;            // Amounts of RGB in original color
	/*0264*/ uint8    unknown0264[2];   // ***Placeholder (Asiel: Has to do with Diety, will unwrap later)
	/*0266*/ uint16   spellId0;         // SpellID of special effect
	/*0268*/ uint16   classes;          // Classes that can use this item
	/*0270*/ uint8    unknown0270[2];   // ***Placeholder
	union
	{
		struct
		{
		/*0272*/ uint16   races;            // Races that can use this item
		/*0274*/ sint8    unknown0274[2];   // ***Placeholder
		/*0276*/ sint8    stackable;        //  1= stackable, 3 = normal, 0 = ? (not stackable)
		} normal;
	};
	/*0277*/ uint8    level;            // Casting level
	union // 0278 has different meanings depending on an stackable
	{
	/*0278*/ sint8    number;          // Number of items in stack
	/*0278*/ int8    charges;         // Number of charges (-1 = unlimited)
	};
	/*0279*/ sint8    effecttype;      // 0=combat, 1=click anywhere w/o class check, 2=latent/worn, 3=click anywhere EXPENDABLE, 4=click worn, 5=click anywhere w/ class check, -1=no effect
		 uint16   spellId;         // spellId of special effect
		 uint8    unknown0282[10]; // ***Placeholder 0288
		 uint32   casttime;        // Cast time of clicky item in miliseconds
		 uint8    unknown0296[16]; // ***Placeholder
		 uint16   skillModId;
		 sint16   skillModPercent;
		 sint16   BaneDMGRace;
		 sint16   BaneDMGBody;
			 // 1 Humanoid, 2 Lycanthrope, 3 Undead, 4 Giant, 5 Construct, 6 Extraplanar, 7 Magical
		 uint8    BaneDMG;
		 uint8    unknown0316[3];
		 uint8    RecLevel;         // max should be 65
		 uint8    RecSkill;         // Max should be 252
		 uint8    unknown0325[2];
		 uint8    ElemDmgType; 
			// 1 Magic, 2 Fire, 3 Cold, 4 Poison, 5 Disease
		 uint8    ElemDmg;
		 uint8    unknown0330[22];
		 uint8    ReqLevel; // Required level
		 uint8    unknown0352[5];
	/*0358*/ int16    focusspellId;
	} common;
	struct // Book Structure (flag == 0x7669)
	{
	/*0228*/ sint8    unknown0172[6];      // ***Placeholder
	/*0234*/ char     file[15];            // Filename of book text on server
	/*0249*/ sint8    unknown0190[15];    // ***Placeholder
	} book;
	struct // containers flag == 0x5400 or 0x5450
	{
		/*0228*/ sint8    unknown0212[41];     // ***Placeholder
		/*0269*/ uint8    numSlots;        // number of slots in container
		/*0270*/ sint8    unknown0214;     // ***Placeholder
		/*0271*/ sint8    sizeCapacity;    // Maximum size item container can hold
		/*0272*/ uint8    weightReduction; // % weight reduction of container
		/*0273*/ uint8    unknown0273[3];     // ***Placeholder
		} container;
};
	inline bool	IsNormal() const		{ return (bool) (type == 0x00); } // ie, not book, not bag
	inline bool	IsBook() const			{ return (bool) (type == 0x02); }
	inline bool	IsBag() const			{ return (bool) (type == 0x01); }
	inline bool	IsStackable() const		{ return (bool) ((type == 0x00) && (common.normal.stackable == 1 || (common.effecttype == 0 && common.normal.stackable == 2))); }
inline bool	IsEquipable(int16 race, int16 class_) const		{
	 if (!this) { /*LogFile->write(EQEMuLog::Debug, "Database error: IsEquipable() called on null item.");*/ return false; }
	 bool israce = false,isclass = false;
	 if (type != 0x00 && equipableSlots == 0) {
	   //LogFile->write(EQEMuLog::Debug, "IsEquipable() type != 0x00 && equipableSlots == 0");
      return false;
     }
     else if (common.classes == 0) {
	   //LogFile->write(EQEMuLog::Debug, "IsEquipable() common.classes == 0");
         return false;
     }
     else if(common.classes == 32767 && common.normal.races == 32767){
         return true;
     }
  else {
     int16 classes_ = common.classes;
     int16 races_ = common.normal.races;

     if(common.classes == 32767) {
	      isclass = true;
     }
     if(common.normal.races == 32767) {
   	      israce = true;
     }
     for (int cur_class = 1; cur_class <= 15; cur_class++) {
        if (classes_ % 2 == 1) {
    	    if(cur_class == class_) {
    	        isclass = true;
	        }
        }
        classes_ = classes_/2;
     }
     for (int cur_race = 1; cur_race <= 14; cur_race++) {
        if (races_ %2 == 1) {
    	    if(cur_race == race) {
    		    israce = true;
   		    }
        }
        races_ = races_/2;
     }
  }
  //LogFile->write(EQEMuLog::Debug, "IsEquipable(%i,%i) israce:%i:%i isclass:%i:%i", race, class_, israce,common.normal.races, isclass, common.classes);
     if(israce && isclass)
          return true;
     else
          return false;
	}
	inline bool	IsExpendable() const	{ return (bool) (common.skill == 21 || common.skill == 11); }
	inline bool	IsGM() const			{ return (bool) (common.GMFlag == -1); }
	inline bool	IsLore() const			{ return (bool) (strstr(lore, "*") != 0); }
	inline bool	IsPendingLore() const	{ return (bool) (strstr(lore, "~") != 0); }
	inline bool	IsArtifact() const		{ return (bool) (strstr(lore, "#") != 0); }
	inline bool IsWeapon() const	{ return (bool) common.damage; }
};

/*
** ZoneServerInfo_Struct
** Zone server information
** Size: 130 bytes
** Used In: OP_ZoneServerInfo
**
*/
struct ZoneServerInfo_Struct
{
/*0000*/	char	ip[128];
/*0128*/	uint16	port;
};

struct CrossServerTell_Struct
{
/*0000*/	char*	ip;
/*0015*/	char	comma;
/*0016*/	int16	port;
/*0021*/	char	comma2;
/*0022*/	char*	servername;
/*0042*/	char	dot;
/*0043*/	char*	name;
/*0107*/	char	comma3;
/*0108*/	char	unknown108[8];
};

struct WhoAllPlayer{
	int32	formatstring;
	int32	pidstring;
	char*	name;
	int32	rankstring;
	char*	guild;
	int32	unknown80[2];
	int32	zonestring;
	int32	zone;
	int32	class_;
	int32	level;
	int32	race;
	char*	account;
	int32	unknown100;
};

struct WhoAllReturnStruct {
	int32	id;
	int32	playerineqstring;
	char	line[27];
	int8	unknown35; //0A
	int32	unknown36;//0s
	int32	playersinzonestring;
	int32	unknown44[2]; //0s
	int32	unknown52;//1
	int32	unknown56;//1
	int32	playercount;//1
	struct WhoAllPlayer player[];
};

struct Trader_Struct {
	int32	code;
	int32	itemid[160];
	int32	unknown;
	int32	itemcost[80];
};

struct ClickTrader_Struct {
	int32	code;
	int32	unknown[161];//damn soe this is totally pointless :/ but at least your finally using memset! Good job :) -LE
	int32	itemcost[80];
};

struct GetItems_Struct{
	int32	items[80];
};

struct BecomeTrader_Struct{
	int32 id;
	int32 code;
};

struct Trader_ShowItems_Struct{
	int32 code;
	int32 traderid;
	int32 unknown08[3];
};

struct TraderBuy_Struct{
	int32 unknown0;
	int32 traderid;
	int32 itemid;
	int32 unknown8;
	int32 price;
	int32 quantity;
	int32 slot_num;
	char  itemname[60];
};

struct TraderItemUpdate_Struct{
	int32 unknown0;
	int32 traderid;
	int8  fromslot;
	int8  toslot; //7?
	int16 charges;
};

struct MoneyUpdate_Struct{
	int32 platinum;
	int32 gold;
	int32 silver;
	int32 copper;
};

struct TraderDelItem_Struct{
	int32 slotid;
	int32 quantity;
	int32 unknown;
};

struct TraderClick_Struct{
	int32 traderid;
	int32 unknown4[2];
	int32 approval;
};

struct FormattedMessage{
	int32	unknown0;
	int32	string_id;
	int32	type;
	char	message[0];
};
struct SimpleMessage_Struct{
	int32	string_id;
	int32	color;
	int32	unknown8;
};
struct GuildMember{
	char	name[64];
	int32	level;
	int32	class_;
	int32	timelaston;
	int32	rank;
	char	publicnote[100];//whatever this is...
	int8	zoneid;
};
struct GuildMember_Struct { //not an eqlive sturct just one for us
	int32 count;
	int32 length;
	struct GuildMember member[];
};
struct GuildMOTD_Struct{
/*0000*/	int32	unknown0;
/*0004*/	char	name[64];
/*0068*/	int32	unknown65[17];
/*0136*/	char	motd[512];
};
struct GuildUpdate_PublicNote{
	int32	unknown0;
	char	name[64];
	char	target[64];
	char	note[100]; //we are cutting this off at 100, actually around 252
};
struct GuildDemoteStruct{
	char	name[64];
	char	target[64];
};
struct GuildRemoveStruct{
	char	target[64];
	char	name[64];
	int32	unknown128;
	int32	leaderstatus; //?
};
struct GuildMakeLeader{
	char	name[64];
	char	target[64];
};
struct BugStruct{
/*0000*/	char	chartype[64];
/*0064*/	char	name[96];
/*0160*/	char	ui[128];
/*0288*/	float	x;
/*0292*/	float	y;
/*0296*/	float	z;
/*0300*/	float	heading;
/*0304*/	int32	unknown304;
/*0308*/	int32	type;
/*0312*/	char	unknown312[2144];
/*2456*/	char	bug[1028];
};
struct Make_Pet_Struct { //Simple struct for getting pet info
	int8 level;
	int8 class_;
	int16 race;
	int8 texture;
	int8 pettype;
	float size;
	int8 type;
};
struct Ground_Spawn{
	float max_x;
	float max_y;
	float min_x;
	float min_y;
	float max_z;
	float heading;
	char name[16];
	int32 item;
	int32 max_allowed;
	int32 respawntimer;
};
struct Ground_Spawns {
	struct Ground_Spawn spawn[50]; //Assigned max number to allow
};
struct PetitionBug_Struct{
	int32	petition_number;
	int32	unknown4;
	char	accountname[64];
	int32	zoneid;
	char	name[64];
	int32	level;
	int32	class_;
	int32	race;
	int32	unknown152[3];
	int32	time;
	int32	unknown168;
	char	text[1028];
};
struct DyeColorStruct{
	int8	blue;
	int8	green;
	int8	red;
	int8	unknown;
};
struct DyeStruct{
	union
	{
		struct
		{
			struct DyeColorStruct head;
			struct DyeColorStruct chest;
			struct DyeColorStruct arms;
			struct DyeColorStruct wrists;
			struct DyeColorStruct hands;
			struct DyeColorStruct legs;
			struct DyeColorStruct feet;
			struct DyeColorStruct unknown;
			struct DyeColorStruct unknown2;
		}dyes;
		struct DyeColorStruct dye[9];
	};
};
struct ApproveZone_Struct {
	char	name[64];
	int32	zoneid;
	int32	approve;
};
struct ZoneInSendName_Struct {
	int32	unknown0;
	char	name[64];
	char	name2[64];
	int32	unknown132;
};
struct ZoneInSendName_Struct2 {
	int32	unknown0;
	char	name[64];
	int32	unknown68[145];
};
// Restore structure packing to default
#pragma pack()

#endif
